//
//  Controller.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "Controller.h"

#import "SBColladaLoader.h"
#import "SBPlyLoader.h"

#import "SBScene.h"
#import "SBMaze.h"
#import "SBEnemy.h"

#import "SBMat4.h"
#import "SBVector3D.h"
#import "SBVector4D.h"
#import "SBNonuniformSpline.h"
#import "SBOctreeNode.h"

#import "SBCamera.h"
#import "SBGLView.h"
#import "SBPathPoints.h"

extern "C" {

#include "LuaObjCBridge/LuaObjCBridge.h"
#include "LuaObjCBridge/lua.h"
#include "LuaObjCBridge/lualib.h"
#include "LuaObjCBridge/lauxlib.h"

}


@implementation Controller

// lua state
static lua_State * LUA;

// our singleton scene pointer
static SBScene * scene;

+ (SBScene *)scene
{
	return scene;
}

static NSMutableDictionary * loadedTextures;

#pragma mark ---- Initialisation ----

- (void) initialiseDefaults
{
	// set the default user prefs in the user defaults database
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	NSArray * values = [NSArray arrayWithObjects:
		[NSNumber numberWithDouble:0.005],
		[NSNumber numberWithBool:NO],
		[NSNumber numberWithBool:YES],
		[NSNumber numberWithBool:NO],
		[NSNumber numberWithBool:NO],
		[NSNumber numberWithBool:NO],
		[NSNumber numberWithInt:2],
		[NSNumber numberWithBool:YES],
		[NSNumber numberWithBool:YES],
		[NSNumber numberWithBool:NO],
		[NSNumber numberWithInt:10],
		nil
	];
	NSArray * keys = [NSArray arrayWithObjects:
		@"renderTimeInterval",
		@"wireframe",
		@"drawinfo",
		@"drawWorldAABBs",
		@"drawLocalAABBs",
		@"drawOctree",
		@"octreeDepth",
		@"drawUsingOctree",
		@"testInFrustum",
		@"drawPathPoints",
		@"mazesize",
		nil
	];
    NSDictionary *appDefaults = [NSDictionary dictionaryWithObjects:values forKeys:keys];
    [defaults registerDefaults:appDefaults];
	
	// initialise our values with the user preferences
	//  - this will use the above defaults unless the user has specified otherwise
	renderTimeInterval = [defaults floatForKey:@"renderTimeInterval"];
	[glView setWireframe:[defaults boolForKey:@"wireframe"]];
	[glView setDrawInfo:[defaults boolForKey:@"drawinfo"]];
	[glView setDrawWorldAABBs:[defaults boolForKey:@"drawWorldAABBs"]];
	[glView setDrawLocalAABBs:[defaults boolForKey:@"drawLocalAABBs"]];
	[glView setDrawOctree:[defaults boolForKey:@"drawOctree"]];
	[SBOctreeNode setMaxDepth:[defaults integerForKey:@"octreeDepth"]];
	[glView setDrawUsingOctree:[defaults boolForKey:@"drawUsingOctree"]];
	[glView setTestInFrustum:[defaults boolForKey:@"testInFrustum"]];
	[glView setDrawPathPoints:[defaults boolForKey:@"drawPathPoints"]];
}

- (void) initialiseMenuItems
{
	[wireframeMenuItem setState:[glView wireframe] ? NSOnState : NSOffState];
	[drawInfoMenuItem setState:[glView drawInfo] ? NSOnState : NSOffState];
	[drawWorldAABBsMenuItem setState:[glView drawWorldAABBs] ? NSOnState : NSOffState];
	[drawLocalAABBsMenuItem setState:[glView drawLocalAABBs] ? NSOnState : NSOffState];
	[drawOctreeMenuItem setState:[glView drawOctree] ? NSOnState : NSOffState];
	[drawUsingOctreeMenuItem setState:[glView drawUsingOctree] ? NSOnState : NSOffState];
	[testInFrustumMenuItem setState:[glView testInFrustum] ? NSOnState : NSOffState];
	[drawPathPointsMenuItem setState:[glView drawPathPoints] ? NSOnState : NSOffState];
}

- (void) awakeFromNib
{		
	// if our glView has been initialised from the nib file
	if( glView != nil )
	{		
		// start our render timer and our 'rotate B' timer
		[self startRenderTimer];
		demo = NO;
		
		// initialise the defaults
		[self initialiseDefaults];
		[self initialiseMenuItems];
		
		// setup lua
		LUA = lua_open();
		if ( LUA == NULL )
		{
			NSLog( @"LUA could not be initialised" );
		}
		lua_baselibopen( LUA );
		
		// run our utilities script
		NSString * bundlePath = [[NSBundle mainBundle] resourcePath];
		lua_dofile( LUA, [[NSString stringWithFormat:@"%@/%s", bundlePath, "utilities.lua"] cString] );
		
		lua_pushstring( LUA, "Controller" );
		lua_objc_pushid( LUA, self);
		lua_settable( LUA, LUA_GLOBALSINDEX );
		
	}
	else
	{
		[self initFailed];
	}
}

- (void) initFailed
{
	NSWindow *infoWindow;
	
	infoWindow = NSGetCriticalAlertPanel( @"Initialization failed",
										  @"Failed to initialize OpenGL",
										  @"OK", nil, nil );
	[NSApp runModalForWindow:infoWindow];
	[infoWindow close];
	[NSApp terminate:self];
}

- (void) dealloc
{	
	if ( cameraSpline )
		[cameraSpline release];
	
	if ( scene )
		[scene release];
	
	if ( loadedTextures )
		[loadedTextures release];
	
	// close lus
	lua_close(LUA);
	
	[super dealloc];
}

#pragma mark ---- Timers ----

- (void) resetRenderTimer
{
	// reseting only has effect if the timer is turned on
	if ( ( renderTimer != nil ) && [renderTimer isValid] )
	{
		[renderTimer invalidate];
		[self startRenderTimer];
	}
}

- (void) startRenderTimer
{
	renderTimer = [[NSTimer scheduledTimerWithTimeInterval:renderTimeInterval
							target:self
							selector:@selector( updateGLView: ) // call this at each interval
							userInfo:nil repeats:YES] retain];
	
	// add to the event and modal run loops
	[[NSRunLoop currentRunLoop] addTimer:renderTimer forMode:NSEventTrackingRunLoopMode];
	[[NSRunLoop currentRunLoop] addTimer:renderTimer forMode:NSModalPanelRunLoopMode];
}

- (void) startDemoTimer
{
	demoTimer = [[NSTimer scheduledTimerWithTimeInterval:0.05
							target:self
							selector:@selector( viewNextFrame: ) // call this at each interval
							userInfo:nil repeats:YES] retain];
	
	// add to the event and modal run loops
	[[NSRunLoop currentRunLoop] addTimer:demoTimer forMode:NSEventTrackingRunLoopMode];
	[[NSRunLoop currentRunLoop] addTimer:demoTimer forMode:NSModalPanelRunLoopMode];
}

// this is called by our render timer
//  - glView is our SBGLView subclassed from NSView, drawRect does all the drawing for us
- (void) updateGLView:(NSTimer *)timer
{
	if( glView != nil )
	{
		[glView drawRect:[glView frame]];
	}
}

- (void)viewNextFrame:(NSTimer *)timer
{
	float nodeInc = 0.06f / [cameraSpline numNodes];
	currentDemoFrame += nodeInc;
	if ( currentDemoFrame > 1 - nodeInc )
	{
		currentDemoFrame = 0.f;
	}
	
	SBCamera * cam = [glView camera];
	SBVector3D * currFramePos = [cameraSpline getPositionAtTime:currentDemoFrame];
	SBVector3D * nextFramePos = [cameraSpline getPositionAtTime:currentDemoFrame + nodeInc];
	[cam setViewPos:currFramePos collidingWithOctree:NO];
	[cam setViewDir:[nextFramePos subtractVector:currFramePos]];
}

#pragma mark ---- IBActions ----

- (IBAction) toggleFullScreen: (id)sender
{
	[glView toggleFullScreen];
}

- (IBAction) toggleWireframe: (id)sender
{
	if ( [glView wireframe] )
	{
		[glView setWireframe:NO];
		[wireframeMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"wireframe"];
	}
	else
	{
		[glView setWireframe:YES];
		[wireframeMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"wireframe"];
	}
}

- (IBAction) toggleDrawInfo: (id)sender
{
	if ( [glView drawInfo] )
	{
		[glView setDrawInfo:NO];
		[drawInfoMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawinfo"];
	}
	else
	{
		[glView setDrawInfo:YES];
		[drawInfoMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawinfo"];
	}
}

- (IBAction) toggleDrawWorldAABBs: (id)sender
{
	if ( [glView drawWorldAABBs] )
	{
		[glView setDrawWorldAABBs:NO];
		[drawWorldAABBsMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawWorldAABBs"];
	}
	else
	{
		[glView setDrawWorldAABBs:YES];
		[drawWorldAABBsMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawWorldAABBs"];
	}
}

- (IBAction) toggleDrawLocalAABBs: (id)sender
{
	if ( [glView drawLocalAABBs] )
	{
		[glView setDrawLocalAABBs:NO];
		[drawLocalAABBsMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawLocalAABBs"];
	}
	else
	{
		[glView setDrawLocalAABBs:YES];
		[drawLocalAABBsMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawLocalAABBs"];
	}
}

- (IBAction) toggleDrawOctree: (id)sender
{
	if ( [glView drawOctree] )
	{
		[glView setDrawOctree:NO];
		[drawOctreeMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawOctree"];
	}
	else
	{
		[glView setDrawOctree:YES];
		[drawOctreeMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawOctree"];
	}
}

- (IBAction) toggleDrawUsingOctree: (id)sender
{
	if ( [glView drawUsingOctree] )
	{
		[glView setDrawUsingOctree:NO];
		[drawUsingOctreeMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawUsingOctree"];
	}
	else
	{
		[glView setDrawUsingOctree:YES];
		[drawUsingOctreeMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawUsingOctree"];
	}
}

- (IBAction) toggleTestInFrustum: (id)sender
{	
	if ( [glView testInFrustum] )
	{
		[glView setTestInFrustum:NO];
		[testInFrustumMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"testInFrustum"];
	}
	else
	{		
		[glView setTestInFrustum:YES];
		[testInFrustumMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"testInFrustum"];
	}
}

- (IBAction) toggleDrawPathPoints: (id)sender
{
	if ( [glView drawPathPoints] )
	{
		[glView setDrawPathPoints:NO];
		[drawPathPointsMenuItem setState:NSOffState];
		[[NSUserDefaults standardUserDefaults] setBool:NO forKey:@"drawPathPoints"];
	}
	else
	{
		[glView setDrawPathPoints:YES];
		[drawPathPointsMenuItem setState:NSOnState];
		[[NSUserDefaults standardUserDefaults] setBool:YES forKey:@"drawPathPoints"];
	}
}

- (IBAction) refreshView: (id)sender
{
	[glView setNeedsDisplay:YES];
}

- (IBAction) resetCamera: (id)sender
{
	if ( glView != nil )
	{
		[[glView camera] resetCamera];
		[glView updateProjection];
	}
}

// toggle the render timer
- (IBAction) toggleRenderTimer: (id)sender
{
	if ( ( renderTimer != nil ) && [ renderTimer isValid ] )
	{
		[ renderTimer invalidate ];
		[ toggleRenderTimerMenuItem setTitle:@"Activate Timer" ];
	}
	else
	{
		[ self startRenderTimer ];
		[ toggleRenderTimerMenuItem setTitle:@"Deactivate Timer" ];	}	
}

// fill with current values and display the preference sheet
- (IBAction) editPreferences: (id)sender
{
	[renderIntervalField setDoubleValue:renderTimeInterval];
	[mazeSizeField setIntValue:[[NSUserDefaults standardUserDefaults] integerForKey:@"mazesize"]];
	[octreeDepthField setIntValue:[[NSUserDefaults standardUserDefaults] integerForKey:@"octreeDepth"]];
	[NSApp beginSheet:preferenceSheet modalForWindow:mainWindow modalDelegate:self
		didEndSelector:NULL contextInfo:nil];
}

// close the preferences sheet without saving
- (IBAction) closePreferenceSheet: (id)sender
{
	[preferenceSheet orderOut:nil];
	[NSApp endSheet:preferenceSheet];
}

// save the preferences and close the sheet
- (IBAction) savePreferences: (id)sender
{
	NSUserDefaults * defaults = [NSUserDefaults standardUserDefaults];
	
	float newRenderTimerInterval = [renderIntervalField floatValue];
	[defaults setFloat:newRenderTimerInterval forKey:@"renderTimeInterval"];	
	renderTimeInterval = newRenderTimerInterval;
	[self resetRenderTimer];
	
	[defaults setInteger:[mazeSizeField intValue] forKey:@"mazesize"];
	
	// if we have a new octreeDepth we must regenerate the octree
	int newDepth = [octreeDepthField intValue];
	[defaults setInteger:newDepth forKey:@"octreeDepth"];
	if ( newDepth != [SBOctreeNode maxDepth] )
	{
		[SBOctreeNode setMaxDepth:newDepth];
		[scene calculateOctree];
	}
	
	[self closePreferenceSheet:self];
}

- (IBAction) openScene: (id)sender
{
	// Display standard open file panel
	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setAllowsMultipleSelection:NO];
	
	NSArray * filesToOpen;
	if ([oPanel runModalForDirectory:@"/Users/spbryson/Documents/"
								file:nil types:[NSArray arrayWithObjects:@"dae", @"ply", nil]] == NSOKButton)
	{
		filesToOpen = [oPanel filenames];
	}
	else
	{
		return;
	}
	
	if ( [filesToOpen count] == 0 )
	{
		return;
	}
	
	if ( scene != nil )
	{
		[scene clear];
	}
	else
	{
		scene = [[SBScene alloc] init];
	}
	
	NSString * filename = [filesToOpen objectAtIndex:0];

	NSString * loadMessage;
	BOOL success;
	if ( [[filename pathExtension] isEqualToString:@"dae"] )
	{
		SBColladaLoader * sceneLoader = [SBColladaLoader alloc];
		success = [sceneLoader loadSceneWithPath:filename intoScene:scene];
		[sceneLoader release];
	}
	else if ( [[filename pathExtension] isEqualToString:@"ply"] )
	{
		SBPlyLoader * plyLoader = [SBPlyLoader alloc];
		success = [plyLoader loadSceneWithPath:filename intoScene:scene];
		[plyLoader release];
	}
	else
	{
		success = NO;
	}
	
	if ( success )
	{
		loadMessage = [NSString stringWithFormat:@"Loaded file: %@", filename];
		
		lua_pushstring( LUA, "Scene" );
		lua_objc_pushid( LUA, scene);
		lua_settable( LUA, LUA_GLOBALSINDEX );
	}
	else
	{
		loadMessage = [NSString stringWithFormat:@"Unable to load file: %@", filename];
	}


	[[NSNotificationCenter defaultCenter] postNotificationName:@"SBMessage"
		object:self userInfo:[NSDictionary dictionaryWithObject:loadMessage forKey:@"message"]];
	
	[scene updateSceneEntityWorldTransforms];
	[scene calculateOctree];
}

// load a new camera spline and call toggleDemo
- (IBAction) openCameraSpline: (id)sender
{
	NSOpenPanel *oPanel = [NSOpenPanel openPanel];
	[oPanel setAllowsMultipleSelection:NO];
	
	NSArray * filesToOpen;
	if ([oPanel runModalForDirectory:@"/Users/spbryson/Documents/"
								file:nil types:[NSArray arrayWithObject:@"spline"]] == NSOKButton)
	{
		filesToOpen = [oPanel filenames];
	}
	else
	{
		return;
	}
	
	if ( [filesToOpen count] == 0 )
	{
		return;
	}
	
	NSDictionary * splineData = [NSDictionary dictionaryWithContentsOfFile:[filesToOpen objectAtIndex:0]];
	NSArray * splineNodes = [splineData objectForKey:@"nodes"];
	
	if ( splineNodes != nil && [splineNodes count] ) // if the spline data is valid
	{
		[[glView camera] resetCamera];
		
		if ( cameraSpline )
		{
			[cameraSpline release];
		}
		cameraSpline = [[SBNonuniformSpline alloc] initWithNodes:splineNodes];
		
		[self toggleDemo:self];
	}
}

// turn demo off/on and load spline if needed
- (IBAction) toggleDemo: (id)sender;
{
	if ( demo )
	{
		demo = NO;
		[demoModeMenuItem setState:NSOffState];
		[demoTimer invalidate];
		[[glView camera] resetCamera];
		
		[[NSNotificationCenter defaultCenter] postNotificationName:@"SBMessage"
			object:self userInfo:[NSDictionary dictionaryWithObject:@"Spline Demo Shutdown" forKey:@"message"]];
	}
	else
	{
		if ( !cameraSpline )
		{
			[self openCameraSpline:self];
		}
		else
		{
			demo = YES;
			[demoModeMenuItem setState:NSOnState];
			[self startDemoTimer];
			
			/*float i;
			for ( i = 0.f; i < 1; i += 0.01 )
			{
				SBVector3D * pos = [cameraSpline getPositionAtTime:i];
				NSLog( @"%f, [ %f, %f, %f ]", i, [pos x], [pos y], [pos z] );
			}*/
		}
		
		[[NSNotificationCenter defaultCenter] postNotificationName:@"SBMessage"
			object:self userInfo:[NSDictionary dictionaryWithObject:@"Started Spline Demo" forKey:@"message"]];
	}
}

- (IBAction) generateMaze: (id)sender
{
	// reset the scene to empty before adding the maze to it
	if ( scene != nil )
	{
		[scene clear];
	}
	else
	{
		scene = [[SBScene alloc] init];
	}
	
	int size = [[NSUserDefaults standardUserDefaults] integerForKey:@"mazesize"];
	SBMaze * maze = [SBMaze alloc];
	[maze initWithSize:size];
	
	[scene setSize:size];
	[scene setCamera:[glView camera]];
	
	// reset the camera spline so the maze can add the spline data
	if ( cameraSpline )
	{
		[cameraSpline release];
	}
	cameraSpline = [[SBNonuniformSpline alloc] init];
	[maze generateMazeWithCameraSpline:cameraSpline];
	//[maze debugMaze];
	
	// add the generated maze to our scene as scene enities
	[maze addMazeToScene:scene];
	[maze setCameraToStartPosition: [glView camera]];
	[scene setPlayerStartLoc: [[glView camera] viewPos]];
	
	// generate our path finding points from the maze and add it to our scene
	SBPathPoints * pathPoints = [[[SBPathPoints alloc] initWithCells:[maze cells]] autorelease];
	[scene setPathPoints: pathPoints];
	
	[scene updateSceneEntityWorldTransforms];
	[scene calculateOctree];
	
	// load the master game entities
	[self loadMasterEntities];
	
	[scene startGame];
	
	lua_pushstring( LUA, "Scene" );
	lua_objc_pushid( LUA, scene);
	lua_settable( LUA, LUA_GLOBALSINDEX );
	
	NSString * message = [NSString stringWithFormat:@"Generated Maze with %dx%d cells", size, size];
	
	[[NSNotificationCenter defaultCenter]
		postNotificationName:@"SBMessage" object:self
					userInfo:[NSDictionary dictionaryWithObject:message forKey:@"message"]];
}

- (void) loadMasterEntities
{
	SBColladaLoader * loader = [[SBColladaLoader alloc] init];
	
	SBSceneEntity * obj = [loader loadObjectWithPath:@"ghost.dae"];
	SBEnemy * ghost = [[[SBEnemy alloc] initWithSceneEntity:obj] autorelease];

	[ghost setCollidable:NO];
	
	[scene setMasterEnemy:ghost];
}

+ (NSDictionary *) lookupGLTexture:(NSString *)imagePath
{
	NSDictionary * lookupEntry = [loadedTextures objectForKey:imagePath];
	if ( lookupEntry )
	{
		return lookupEntry;
	}
	
	return nil;
}

+ (GLuint) loadGLTexture:(NSString *)imagePath
{
	
	if ( loadedTextures == nil )
	{
		loadedTextures = [[NSMutableDictionary alloc] init];
	}
	
	//NSLog( @"Loading texture: %@", imagePath );
	NSDictionary * lookupEntry = [self lookupGLTexture:imagePath];
	if ( lookupEntry != nil )
	{
		return (GLuint)[[lookupEntry objectForKey:@"id"] unsignedIntValue];
	}
	
	NSSize imageSize;
	NSData *data;
	GLuint i;
	
	if ( data = [NSData dataWithContentsOfFile: imagePath] )
	{
		GLuint textureIdx;
		glGenTextures( 1, &textureIdx );
		
		glBindTexture( GL_TEXTURE_2D, textureIdx );
		
		// Linear filtering
		glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
		glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
		
		NSBitmapImageRep *imageRep = [NSBitmapImageRep imageRepWithData:data];
		
		GLenum mode;
		if ( [imageRep samplesPerPixel] == 4 )
		{
			mode = GL_RGBA;
		}
		else if ( [imageRep samplesPerPixel] == 3)
		{
			mode = GL_RGB;
		}
		else if ( [imageRep samplesPerPixel] == 2)
		{
			mode = GL_LUMINANCE_ALPHA;
		}
		else
		{
			mode = GL_LUMINANCE;
		}
		
		GLubyte *image = [imageRep bitmapData];
		GLuint samples = [imageRep samplesPerPixel];
		imageSize = [imageRep size];
		
		GLubyte *data;
		data = (GLubyte *)malloc( sizeof(GLubyte) * (GLuint)( imageSize.width * imageSize.height * samples ));
		
		// NSBitmap rep is flipped compared to the GL coordinate system
		//  - so we need to flip it back
		for ( i = 0; i < imageSize.height; ++i )
		{
			memcpy( data + (GLuint)( ( imageSize.height - 1 - i ) * imageSize.width * samples ),
					image + (UInt32)( i * imageSize.width * samples ), (UInt32)( imageSize.width * samples ) );
		}
		
		glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, (GLsizei)( imageSize.width ), (GLsizei)( imageSize.height ),
					  0, mode, GL_UNSIGNED_BYTE, data );
		
		free(data);
		
		NSDictionary * texEntry = [NSDictionary dictionaryWithObjects:
			[NSArray arrayWithObjects:
				[NSNumber numberWithUnsignedInt:textureIdx],
				[NSNumber numberWithFloat:imageSize.width],
				[NSNumber numberWithFloat:imageSize.height],
				nil]
															  forKeys:
			[NSArray arrayWithObjects:
				@"id",
				@"width",
				@"height",
				nil]];
		
		[loadedTextures setObject:texEntry forKey:imagePath];
		
		return textureIdx;
	}
	
	NSLog(@"Failed to load image %@", imagePath);
	return 0;
}


- (IBAction) executeLua: (id)sender
{
	NSString * cmd = [[luaConsoleInputText string] retain];
	
	int result = lua_dostring( LUA, [cmd cString] );
	if ( result == LUA_ERRSYNTAX )
	{
		NSLog( @"LUA_ERRSYNTAX" );
	}
	else if ( result == LUA_ERRMEM )
	{
		NSLog( @"LUA_ERRMEM" );
	}
}

@end