//
//  SBCamera.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBCamera.h"
#import "SBVector3D.h"
#import "AABB.h"
#import "SBOctreeNode.h"
#import "SBScene.h"
#import "Controller.h"

@implementation SBCamera

- (id) init
{
	if ( self = [super init] )
	{
		active = NO;
		
		totalFrames = 0;
		viewPos = [[SBVector3D alloc] init];
		viewDir = [[SBVector3D alloc] init];
		viewUp = [[SBVector3D alloc] init];
		rotPoint = [[SBVector3D alloc] init];
		localAABB = [[AABB alloc] initWithMin:[SBVector3D vector3DWithValuesX:-0.15 y:-0.3 z:-0.15]
									      max:[SBVector3D vector3DWithValuesX: 0.15 y: 0.15 z: 0.15]];
		worldAABB = [[AABB alloc] initWithAABB:localAABB];

		return self;
	}
	return nil;
}

- (void) dealloc
{
	[viewPos release];
	[viewDir release];
	[viewUp release];
	[rotPoint release];
	[localAABB release];
	[worldAABB release];
	
	if ( positionControlPoints != nil )
	{
		[positionControlPoints release];
	}
	
	[super dealloc];
}

- (void) setActive:(BOOL)_active { active = _active; }

// sets the camera data to initial conditions
- (void) resetCamera
{
   [viewPos setWithValuesX:0.0 y:0.0 z:-10.0];
   [viewDir setWithValuesX:0.0 y:0.0 z:  1.0];
   [viewUp  setWithValuesX:0.0 y:1.0 z:  0.0];
   [rotPoint setWithValuesX:0.0 y:0.0 z: 0.0];
   
   aperture = 40;
   worldRotation[0] = worldRotation[1] = worldRotation[2] = worldRotation[3] = 0.0f;
}

- (void) setViewPos:(SBVector3D *)newViewPos collidingWithOctree:(BOOL)collide
{
	AABB * newAABB = [[AABB alloc] initWithAABB:localAABB];
	[newAABB translate:newViewPos];
	
	SBScene * scene = [Controller scene];
	
	if ( [scene rootOctreeNode] )
	{
		// be careful here, using this code we can only access the first object we are
		//  - colliding with, even if we are colliding with more
		//  - this is sufficient for this game however
		SBSceneEntity * collidedWithEntity = [[scene rootOctreeNode] collideWith:newAABB];
		
		if ( collide && collidedWithEntity )
		{
			// collidable objects are walls and floors etc
			if ( [collidedWithEntity collidable] )
			{
				return;
			}
			
			// we have collided with something that is not a wall or floor
			// lets let the scene figure out what to do
		}
	}
	
	[viewPos assign:newViewPos];
	[worldAABB assign:newAABB];
}

- (void) setViewDir:(SBVector3D *)newViewDir { [viewDir assign:newViewDir]; }
- (void) setViewUp:(SBVector3D *)newViewUp { [viewUp assign:newViewUp]; }

- (void) setHeight:(GLint)newViewHeight { viewHeight = newViewHeight; }
- (void) setWidth:(GLint)newViewWidth { viewWidth = newViewWidth; }
- (void) setAperture:(GLdouble)newAperture { aperture = newAperture; }

- (SBVector3D *) viewPos { return viewPos; }
- (SBVector3D *) viewDir { return viewDir; }
- (SBVector3D *) viewUp { return viewUp; }
- (SBVector3D *) rotPoint { return rotPoint; }

- (GLdouble) aperture { return aperture; }
- (GLint) viewHeight { return viewHeight; }
- (GLint) viewWidth { return viewWidth; }
- (GLfloat *) worldRotation { return worldRotation; }
- (AABB *) localAABB { return localAABB; }
- (AABB *) worldAABB { return worldAABB; }

- (void) calculateWorldAABB
{
	[worldAABB assign:localAABB];
	[worldAABB translate:viewPos];
}

// navigation
- (void) move:(int)dir
{
	if ( !active )
		return;
	
	SBVector3D * direction = [[[[SBVector3D alloc] initWithVector3D:viewDir] unit] divideByFloat:10];
	SBVector3D * newPosition;
	if ( dir )
	{
		newPosition = [[viewPos addVector:direction] retain];
	}
	else
	{
		newPosition = [[viewPos subtractVector:direction] retain];
	}
	[self setViewPos:newPosition collidingWithOctree:YES];
	[newPosition release];
}

- (void) elevate:(int)dir
{
	if ( !active )
		return;
	
	SBVector3D * direction = [[SBVector3D vector3DWithValuesX:0 y:0.1 z:0] retain];
	SBVector3D * newPosition;
	if ( dir )
	{
		newPosition = [[viewPos addVector:direction] retain];
	}
	else
	{
		newPosition = [[viewPos subtractVector:direction] retain];
	}
	[self setViewPos:newPosition collidingWithOctree:YES];
	[newPosition release];
}

- (void) strafe:(int)dir
{
	if ( !active )
		return;
	
	SBVector3D * direction = [[SBVector3D alloc] initWithVector3D:viewDir];
	SBVector3D * newPosition;
	[direction assign:[[[direction crossProduct:viewUp] unit] divideByFloat:10]];
	if ( dir )
	{
		newPosition = [[viewPos subtractVector:direction] retain];
	}
	else
	{
		newPosition = [[viewPos addVector:direction] retain];
	}
	[self setViewPos:newPosition collidingWithOctree:YES];
	[newPosition release];
}

- (void) turn:(int)dir
{
	if ( !active )
		return;
	
	[self turnWithRotation: dir ? -0.1 : 0.1];
}

- (void) turnWithRotation:(float)angle
{
	if ( !active )
		return;
	
	float cosAng = cos( angle );
	float sinAng = sin( angle );
	float newX = [viewDir x] * cosAng - [viewDir z] * sinAng;
	float newZ = [viewDir z] * cosAng + [viewDir x] * sinAng;
	
	[viewDir setX:newX];
	[viewDir setZ:newZ];
}


@end
