//
//  SBColladaLoader.mm
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBColladaLoader.h"

#import "SBStream.h"
#import "SBDrawable.h"

#import "SBScene.h"
#import "SBSceneEntity.h"

#import "AABB.h"
#import "SBMat4.h"
#import "SBVector3D.h"
#import "SBVector4D.h"

#include "tinyxml.h"
#include "DaeSyntax.h"

#import "Controller.h"

TiXmlElement * root;

@implementation SBColladaLoader

static TiXmlElement * getChildWithAttr( TiXmlElement * _parent, const char * _tag,
											const char * _attrName, const char * _attrValue )
{
	for ( TiXmlNode * i = _parent->FirstChild( _tag ); i; i = _parent->IterateChildren( _tag, i ) )
	{
		TiXmlElement * child = i->ToElement();
		if ( !child )
		{
			continue;
		}
		const char * attributeValue = child->Attribute( _attrName );
		if ( attributeValue && strcmp( attributeValue, _attrValue ) == 0 )
		{
			return child;
		}
	}
	return 0;
}


static TiXmlElement * getGeometryNode( const char * _url )
{
	// find any library tags with attribute type="GEOMETRY"
	// if we find a geom node with an id == _url then we have found our source geom node
	
	for ( TiXmlNode * i = root->FirstChild( COLLADA_LIBRARY_STRUCTURE ); i;
					i = root->IterateChildren( COLLADA_LIBRARY_STRUCTURE, i ) )
	{
		TiXmlElement * lib = i->ToElement();
		if ( !lib )
		{
			continue;
		}
		
		if ( strcmp( lib->Attribute( COLLADA_TYPE_PROPERTY ), COLLADA_GEOMETRY_LIBRARY_TYPE ) == 0 )
		{
			// find any geometry tags
			return getChildWithAttr( lib, COLLADA_GEOMETRY_STRUCTURE, COLLADA_ID_PROPERTY, _url );
		}
	}
	return 0;
}

static void readFloats( const char * _v, int _count, float * _ret, float _scale )
{
	for ( int i = 0; i < _count; ++i )
	{
		while ( isspace( *_v ) )
		{
			++_v;
		}
		
		*_ret = ( (float) atof( _v )  * _scale );
		_ret++;
		
		// Advance to the next v
		++_v;
		while ( *_v && !isspace( *_v ) )
		{
			++_v;
		}
	}
}

static int getStreamSize( SBStreamId _streamType )
{
	if ( _streamType == SI_Position ||
		 _streamType == SI_Normal ||
		 _streamType == SI_Tangent ||
		 _streamType == SI_Binormal )
	{
		return 3;
	}
	else if ( _streamType == SI_UV0 ||
			  _streamType == SI_UV1 ||
			  _streamType == SI_UV2 ||
			  _streamType == SI_UV3 )
	{
		return 2;
	}
	assert ( "Unhandled stream type" );
	return -1;
}

// this relies on the matrix not being decomposed... check the collada file
// in the future we can add support for decomposed matrices
// need to handle the scale in the collada file
- (SBMat4 *) getTransformFromNode:(TiXmlNode *)_node
{
	TiXmlNode * matrixNode = _node->FirstChild( COLLADA_MATRIX_STRUCTURE );
	
	if ( matrixNode )
	{
		float m[16];
		readFloats( matrixNode->FirstChild()->Value(), 16, m, 1 );
		SBMat4 * mat = [SBMat4 mat4WithVec4Col0:[SBVector4D vector4DWithValuesX:m[ 0] y:m[ 1] z:m[ 2] w:m[ 3]]
						   col1:[SBVector4D vector4DWithValuesX:m[ 4] y:m[ 5] z:m[ 6] w:m[ 7]]
						   col2:[SBVector4D vector4DWithValuesX:m[ 8] y:m[ 9] z:m[10] w:m[11]]
						   col3:[SBVector4D vector4DWithValuesX:m[12] y:m[13] z:m[14] w:m[15]]];
		
		[mat transpose];
		return mat;
	}
	return nil;
}

- (AABB *) getAABBFromNode:(TiXmlNode *)_node
{
	TiXmlNode * bbNode = _node->FirstChild( COLLADA_BOUNDINGBOX_STRUCTURE );
	
	if ( bbNode )
	{
		AABB * aabb = [AABB alloc];
		
		TiXmlNode * minNode = bbNode->FirstChild( COLLADA_MIN_STRUCTURE );
		TiXmlNode * maxNode = bbNode->FirstChild( COLLADA_MAX_STRUCTURE );

		float min[3], max[3];
		readFloats( minNode->FirstChild()->Value(), 3, min, 1 );
		readFloats( maxNode->FirstChild()->Value(), 3, max, 1 );
		
		[aabb initWithMin:[SBVector3D vector3DWithValuesX:min[0] y:min[1] z:min[2]]
					  max:[SBVector3D vector3DWithValuesX:max[0] y:max[1] z:max[2]]];
		return aabb;
	}
	
	return nil;
}

- (SBVertexStream *) parseVertices:(TiXmlNode *)vertsArrayNode
{
	assert( vertsArrayNode );
	TiXmlElement * vertsArray = vertsArrayNode->ToElement();
	assert( vertsArray );
	
	// create a VertexStream to store the verts in
	int count = 0;
	vertsArray->QueryIntAttribute( COLLADA_COUNT_PROPERTY, &count );
	assert( count );
	SBVertexStream * pos = [[[SBVertexStream alloc] initWithNumElements:count/3
													stride:3 streamType:SI_Position] autorelease];
	
	readFloats( vertsArrayNode->FirstChild()->Value(), count, [pos data], 1 );
	return pos;
}

// getVerts will add to global stream cache
- (SBVertexStream *) getVerticesForMesh:(TiXmlElement *)_mesh withId:(const char *)_sourceID
{
	// Have we seen this before?
	SBVertexStream * foundStream = [globalStreamsCache objectForKey:
										[NSString stringWithCString:_sourceID]];
	
	if ( foundStream != nil )
	{
		return foundStream;
	}
	
	// search _mesh for <vertices> tag with _sourceID
	TiXmlElement * verts = getChildWithAttr( _mesh, COLLADA_VERTICES_STRUCTURE,
													COLLADA_ID_PROPERTY, _sourceID );
	assert( verts && "Could not find vertices tag" );
	
	
	// find <input> tag with semantic POSITION and get positionSource
	TiXmlElement * input = getChildWithAttr( verts, COLLADA_INPUT_STRUCTURE,
											 COLLADA_SEMANTIC_PROPERTY, COLLADA_POSITION_INPUT );
	assert( input && "Could not find input tag" );
	
	const char * vertsSource = input->Attribute( COLLADA_SOURCE_STRUCTURE );
	++vertsSource; // remove the # from the front of the source string
	
	// search _mesh for <source> tag with positionSource
	TiXmlElement * source = getChildWithAttr ( _mesh, COLLADA_SOURCE_STRUCTURE,
												COLLADA_ID_PROPERTY, vertsSource );
	assert( source && "Could not find verts source");
	
	// should now check the <technique> tag for the array id but we are going to assume
	// the first array is the one we want
	TiXmlNode * vertsArrayNode = source->FirstChild( COLLADA_FLOAT_ARRAY_STRUCTURE );
	assert( vertsArrayNode && "Could not find COLLADA float array structure for stream" );
	
	SBVertexStream * pos = [self parseVertices:vertsArrayNode];
	//[pos debug:@"Vertices"];
	
	[globalStreamsCache setObject:pos forKey:[NSString stringWithCString:_sourceID]];
	return pos;
}

- (SBVertexStream *) getStreamForMesh:(TiXmlElement *)_mesh withId:(const char *)_sourceID
						   streamType:(SBStreamId)_streamType
{
	// Have we seen this before?
	SBVertexStream * foundStream = [globalStreamsCache objectForKey:
		[NSString stringWithCString:_sourceID]];
	
	if ( foundStream != nil )
	{
		return foundStream;
	}
	
	// search _mesh for <source> tag with _sourceID
	TiXmlElement * source = getChildWithAttr( _mesh, COLLADA_SOURCE_STRUCTURE,
													COLLADA_ID_PROPERTY, _sourceID );
	assert( source && "Could not find stream source" );
	
	// should now check the <technique> tag for the array id but we are going to assume
	// the first array is the one we want
	TiXmlNode * arrayNode = source->FirstChild( COLLADA_FLOAT_ARRAY_STRUCTURE );
	assert( arrayNode && "Could not find COLLADA float array structure for stream" );
	
	TiXmlElement * array = arrayNode->ToElement();
	
	// create a VertexStream
	int count = 0;
	array->QueryIntAttribute( COLLADA_COUNT_PROPERTY, &count );
	assert( count && "Could not get count for array structure" );
	int stride = getStreamSize( _streamType );
	
	SBVertexStream * stream = [[[SBVertexStream alloc] initWithNumElements:count/stride
												stride:stride streamType:_streamType] autorelease];
	
	readFloats( arrayNode->FirstChild()->Value(), count, [stream data], 1 );
	
	[globalStreamsCache setObject:stream forKey:[NSString stringWithCString:_sourceID]];
	
	//[stream debug:[NSString stringWithFormat:@"for mesh %s", _sourceID]];
	
	return stream;
}

// cache streams for this mesh and store in a local mesh streams map
- (void) cacheStreamsFromMesh:(TiXmlElement *)_mesh
				  intoStreamsCache:(NSMutableDictionary *)_meshStreamsCache
{
	SBVertexStream * currStream;
	
	for ( TiXmlNode * j = _mesh->FirstChild( COLLADA_POLYGONS_STRUCTURE ); j;
					j = _mesh->IterateChildren( COLLADA_POLYGONS_STRUCTURE, j ) )
	{
		TiXmlElement * polygons = j->ToElement();
		if ( !polygons )
		{
			continue;
		}
		
		// find vertex and norm array
		for ( TiXmlNode * i = polygons->FirstChild( COLLADA_INPUT_STRUCTURE ); i;
						i = polygons->IterateChildren( COLLADA_INPUT_STRUCTURE, i ) )
		{
			
			TiXmlElement * input = i->ToElement();
			if ( !input )
			{
				continue;
			}
			
			const char * source = input->Attribute( COLLADA_SOURCE_STRUCTURE );
			assert( source && "No source attribute found" );
			++source; // remove the # from the front of the string
			
			const char* semantic_property = input->Attribute( COLLADA_SEMANTIC_PROPERTY );
			
			if ( strcmp( semantic_property, COLLADA_VERTEX_INPUT ) == 0 )
			{
				currStream = [self getVerticesForMesh:_mesh withId:source];
			}
			else if ( strcmp( semantic_property, COLLADA_NORMAL_INPUT ) == 0 )
			{
				currStream = [self getStreamForMesh:_mesh withId:source streamType:SI_Normal];
			}
			else if ( (strcmp(semantic_property, COLLADA_MAPPING_INPUT)     == 0) ||
					  (strcmp(semantic_property, COLLADA_ALT_MAPPING_INPUT) == 0) ||
					  (strcmp(semantic_property, COLLADA_TEXCOORD_INPUT)    == 0) )
			{
				// all cached collada UV streams use id SI_UV0
				currStream = [self getStreamForMesh:_mesh withId:source streamType:SI_UV0];
			}
			else
			{
				assert(0 && "Unknown semantic property for stream");
			}
			
			[_meshStreamsCache setObject:currStream forKey:[NSString stringWithCString:source]];
		}
	}
}

- (int) prepareDrawableStreamsWithPolygons:(TiXmlElement *)_polygons
						  meshStreamsCache:(NSDictionary *)_meshStreamsCache
							colladaStreams:(NSMutableArray *)_cldStreams
						   drawableStreams:(NSMutableArray *)_drwStreams
							drawableCaches:(NSMutableArray *)_drwCaches
{
	int currUVStream = 0;
	NSMutableDictionary * sourceTags = [[NSMutableDictionary alloc] init];
	
	// for each input tag, find the corresponding stream in _meshStreams and _drawableStreams
	for ( TiXmlNode * i = _polygons->FirstChild( COLLADA_INPUT_STRUCTURE ); i;
						i = _polygons->IterateChildren( COLLADA_INPUT_STRUCTURE, i ) )
	{
		TiXmlElement * input = i->ToElement();
		assert( input && "TinyXML error. Expected element from node" );
		
		// find the cached stream
		const char * source = input->Attribute( COLLADA_SOURCE_STRUCTURE );
		assert( source && "No source attribute found" );
		++source; // remove the # from the front of the string
		
		int idx = -1;
		input->QueryIntAttribute( COLLADA_IDX_PROPERTY, &idx );
		assert ( idx >= 0 && "Invalid idx value" );
		
		// get the idx key, if it is not in the dictionary then prepare the stream
		if ( ![sourceTags objectForKey:[NSNumber numberWithInt:idx]] )
		{
			// assert that we have a stream with this source in our meshStreamsCache
			SBVertexStream * cachedStream = [_meshStreamsCache objectForKey:[NSString stringWithCString:source]];
			assert( cachedStream && "Expected stream not found in meshStreamsCache" );
			
			// add this stream to our collada streams
			[_cldStreams addObject:cachedStream];
			
			// create the stream that we are going to add to our drawable later
			SBVertexStream * newStream = [[[SBVertexStream alloc] init] autorelease];
			[newStream setStride:[cachedStream stride]];
			if ( [cachedStream streamType] >= SI_UV0 && [cachedStream streamType] <= SI_UV3 )
			{
				[newStream setStreamType:(SBStreamId)(SI_UV0 + currUVStream)];
				currUVStream++;
			}
			else
			{
				[newStream setStreamType:[cachedStream streamType]];
			}
			
			// add the new stream to our drawable streams array
			[_drwStreams addObject:newStream];
				
			// create a draw caches for this stream
			NSMutableArray * drawableCache = [[[NSMutableArray alloc] init] autorelease];
			[_drwCaches addObject:drawableCache];
			
			// add this source to our dictionary of sources
			[sourceTags setObject:[NSString stringWithCString:source]
						   forKey:[NSNumber numberWithInt:idx]];
		}
	}
	return [sourceTags count];
}

- (void) readIntsFromString:(const char *)_str into:(NSMutableArray *)_array
{
	const char * current = _str;
	while( *current )
	{
		char* next;
		long value = strtol( current, &next, 10 );
		if( current != next )
		{
			[_array addObject:[NSNumber numberWithLong:( value )]];
			current = next;
		}
		else
		{
			++current;
		}
	}
}

- (int) findOrAddToStreamWithValues:(float *[])_values
					 drawableCaches:(NSMutableArray *)_drwCaches
						streamSizes:(int[])_streamSizes
{	
	unsigned int numElements = [[_drwCaches objectAtIndex:0] count] / _streamSizes[0];
	
	// not found so we add it here
	for ( unsigned int i = 0; i < [_drwCaches count]; ++i )
	{
		NSMutableArray * currentDrawableCache = [_drwCaches objectAtIndex:i];
		for (int j = 0; j < _streamSizes[i]; ++j )
		{
			[currentDrawableCache addObject: [NSNumber numberWithFloat: _values[i][j] ] ];
		}
	}
	
	return numElements; // Use the old size ( newsize - 1: last element ).
}


- (void) addToDrawable:(SBDrawable *)_d withPolygons:(TiXmlElement *)_polygons
								meshStreamsCache:(NSDictionary *)_meshStreamsCache
{
	NSMutableArray * cldStreams = [[NSMutableArray alloc] init];
	NSMutableArray * drwStreams = [[NSMutableArray alloc] init];
	NSMutableArray * drwCaches = [[NSMutableArray alloc] init];
	
	int index_stride = [self prepareDrawableStreamsWithPolygons:_polygons
										   meshStreamsCache:_meshStreamsCache
											 colladaStreams:cldStreams
											drawableStreams:drwStreams
											 drawableCaches:drwCaches];
	
	/*NSEnumerator * en = [cldStreams objectEnumerator];
	id obj;
	while( obj = [en nextObject] )
	{
		[obj debug:@"cldStream"];
	}*/
	
	const int NumberOfStreams = 8;
	NSMutableArray * drawableIndexData = [[NSMutableArray alloc] init];
	
	// iterate through all <p> tags
	for ( TiXmlNode * i = _polygons->FirstChild( COLLADA_POLYGON_STRUCTURE ); i;
						i = _polygons->IterateChildren( COLLADA_POLYGON_STRUCTURE, i ) )
	{
		// read the indicies into an array
		NSMutableArray * polygonIndexData = [[NSMutableArray alloc] init];
		const char * indiciesString = i->FirstChild()->Value();
		[self readIntsFromString:indiciesString into:polygonIndexData];
		
		// debug the index data
		/*NSMutableString * element = [NSMutableString stringWithString:@"Polygon indicies: "];
		for ( unsigned int j = 0; j < [polygonIndexData count]; ++j )
		{
			[element appendString:[NSString stringWithFormat:@"%d, ", [[polygonIndexData objectAtIndex:j] intValue]]];
		}
		NSLog( element );
		NSLog( @"\n\n" );*/
		
		// store the strides of each cld stream elements in an array (eg 4 for position, 3 for normal).
		int streamSizes[ NumberOfStreams ];
		for ( int j = 0; j < [cldStreams count]; ++j )
		{
			streamSizes[j] = [[cldStreams objectAtIndex:j] stride];
			assert(streamSizes[j] > 0 && streamSizes[j] < 16);
		}
		
		// first generate the indicies from the <p> data tag
		int cldIdxA[ NumberOfStreams ];
		for ( int j = 0; j < index_stride; ++j )
		{
			cldIdxA[j] = [[polygonIndexData objectAtIndex:j] intValue];
		}
		
		// look up the value for each index
		float * cldValA[ NumberOfStreams ];
		for ( int j = 0; j < [cldStreams count]; ++j )
		{
			int numFloats = [[cldStreams objectAtIndex:j] stride];
			SBVertexStream * cldStream = [cldStreams objectAtIndex:j];
			cldValA[j] = &( [cldStream data][ numFloats * cldIdxA[j] ] );
		}
		
		// push these values onto our drwCaches
		int drwIdxA = [self findOrAddToStreamWithValues:cldValA
										 drawableCaches:drwCaches
											streamSizes:streamSizes];
		
		// triangulate and generate the index stream
		int currTri = 1;
		int remainingVerts = [polygonIndexData count] / index_stride;
		
		while ( remainingVerts >= 3 )
		{
			int cldIdxB[ NumberOfStreams ], cldIdxC[ NumberOfStreams ];
			for ( int j = 0; j < index_stride; ++j )
			{
				cldIdxB[j] = [[polygonIndexData objectAtIndex:(index_stride * currTri) + j] intValue]; // second vert
				cldIdxC[j] = [[polygonIndexData objectAtIndex:(index_stride * currTri) + index_stride + j] intValue]; // third vert
			}
			
			// get the value of each Cld stream entry
			float * cldValB[ NumberOfStreams ];
			float * cldValC[ NumberOfStreams ];
			
			for (int j = 0; j < [cldStreams count]; ++j )
			{
				int numFloats = [[cldStreams objectAtIndex:j] stride];
				SBVertexStream * cldStream = [cldStreams objectAtIndex:j];
				cldValB[j] = &( [cldStream data][numFloats * cldIdxB[j]] );
				cldValC[j] = &( [cldStream data][numFloats * cldIdxC[j]] );
			}
						
			// push vertex b, c values onto our drwCaches and add their indicies
			int drwIdxB = [self findOrAddToStreamWithValues:cldValB
											 drawableCaches:drwCaches
												streamSizes:streamSizes];
			
			int drwIdxC = [self findOrAddToStreamWithValues:cldValC
											 drawableCaches:drwCaches
												streamSizes:streamSizes];
			
			// add vertex a to our index data
			[drawableIndexData addObject:[NSNumber numberWithInt:drwIdxA]];
			[drawableIndexData addObject:[NSNumber numberWithInt:drwIdxB]];
			[drawableIndexData addObject:[NSNumber numberWithInt:drwIdxC]];
			
			currTri++;
			remainingVerts--;
		}
	}
	
	// debug the draw caches
	/*for ( unsigned int i = 0; i < [drwCaches count]; ++i )
	{
		NSArray * currDrwCache = [drwCaches objectAtIndex:i];
		unsigned int stride = [[drwStreams objectAtIndex:i] stride];
		NSLog( @"Draw Cache" );
		for ( unsigned int j = 0; j < [currDrwCache count]; j += stride )
		{
			NSMutableString * element = [NSMutableString stringWithString:@"Element: "];
			for ( unsigned int k = 0; k < stride; ++k )
			{
				[element appendString:[NSString stringWithFormat:@"%f, ", [[currDrwCache objectAtIndex:j+k] floatValue]]];
			}
			NSLog( element );
		}
		NSLog( @"\n\n" );
	}*/
	
	// set up drawable indicies
	SBIndexStream * drwIdxStream = [[SBIndexStream alloc] initWithNumElements:[drawableIndexData count]];
	SBVertexStream * positionStream = [drwStreams objectAtIndex:0]; // this is assuming the pos index is 0
	NSArray * positionDrawCache = [drwCaches objectAtIndex:0];
	[_d setNumVertices:[positionDrawCache count] / [positionStream stride]];
	unsigned short * drwIdxData = [drwIdxStream data];
	
	// copy the data from drawableIndexData into drwIdxStream
	for ( unsigned int i = 0; i < [drawableIndexData count]; ++i )
	{
		assert ([[drawableIndexData objectAtIndex:i] intValue]  < 65536 ); // because index data is unsigned short
		drwIdxData[i] = [[drawableIndexData objectAtIndex:i] shortValue];
	}
	
	//[drwIdxStream debug:@"Drawable Index Stream"];
	
	[_d setIndexStream:drwIdxStream];
	
	// create and add the streams to d
	for ( unsigned int i = 0; i < [drwCaches count]; ++i )
	{
		NSMutableArray * currDrawableCache = [drwCaches objectAtIndex:i];
		
		// using the drawableCache, initialise the drawableStreams that have already been prepared
		SBVertexStream * currDrawableStream = [drwStreams objectAtIndex:i];
		[currDrawableStream setNumElements: [currDrawableCache count] / [[drwStreams objectAtIndex:i] stride]];
		[currDrawableStream allocData];
		float * currDrawStreamData = [currDrawableStream data];
		
		// copy the data
		for (unsigned int j = 0; j < [currDrawableCache count]; ++j )
		{
			currDrawStreamData[j] = [[currDrawableCache objectAtIndex:j] floatValue];
		}
		
		[_d addStream: currDrawableStream];
	}
	
	/*en = [drwStreams objectEnumerator];
	while( obj = [en nextObject] )
	{
		[obj debug:@"drwStream"];
	}*/

	[drwCaches release];
}


// using the node and transform info already obtained, this loads all the verts, uvs etc into a MeshMap
- (void) addGeometry:(TiXmlElement *)_geomNode toSceneEntity: (SBSceneEntity *)_sceneEntity
{
	const char * geomId = _geomNode->Attribute( COLLADA_ID_PROPERTY );
	assert( geomId && "Could not get the geomId attribute" );
	
	for ( TiXmlNode * i = _geomNode->FirstChild( COLLADA_MESH_STRUCTURE ); i;
					i = _geomNode->IterateChildren( COLLADA_MESH_STRUCTURE, i ) )
	{
		TiXmlElement * mesh = i->ToElement();
		if ( !mesh )
		{
			continue;
		}
		
		NSMutableDictionary * meshStreamsCache = [[NSMutableDictionary alloc] init];
		// cache all streams for this mesh
		[self cacheStreamsFromMesh:mesh intoStreamsCache:meshStreamsCache];
		
		for ( TiXmlNode * j = mesh->FirstChild( COLLADA_POLYGONS_STRUCTURE ); j;
						j = mesh->IterateChildren( COLLADA_POLYGONS_STRUCTURE, j ) )
		{
			TiXmlElement * polygons = j->ToElement();
			if ( !polygons )
			{
				continue;
			}

			SBDrawable * d = [[SBDrawable alloc] init];
						
			const char * material = polygons->Attribute( COLLADA_MATERIAL_PROPERTY );
			if ( material )
			{
				NSString * bundlePath = [[NSBundle mainBundle] resourcePath];
				
				unsigned int textureIdx = [Controller loadGLTexture:
					[NSString stringWithFormat:@"%@/%s", bundlePath, material]];
				[d setTextureIdx:textureIdx];
			}
			
			const char * material2 = polygons->Attribute( COLLADA_ALT_MATERIAL_PROPERTY );
			if ( material2 )
			{
				NSString * bundlePath = [[NSBundle mainBundle] resourcePath];
				
				unsigned int textureIdx = [Controller loadGLTexture:
					[NSString stringWithFormat:@"%@/%s", bundlePath, material2]];
				[d setAlternateTextureIdx:textureIdx];
			}

			[self addToDrawable:d withPolygons:polygons meshStreamsCache:meshStreamsCache];
			//[[d streamWithType:SI_Position] debug:@"Vertex Stream"];
			//[[d indexStream] debug:@"Index Stream"];

			[_sceneEntity addDrawable:d];
			[d release];
		}
		
		[meshStreamsCache release];
	}
}

- (SBSceneEntity *) processNode:(TiXmlElement *)_node parentEntity:(SBSceneEntity *)_parentEntity;
{
	// create our scene entity to represent this node
	SBSceneEntity * entity = [[SBSceneEntity alloc] init];
	
	// set the transform of this scene entity
	SBMat4 * transform = [self getTransformFromNode:_node];
	if ( transform )
	{
		[entity setLocalTransform:transform];
	}
	
	AABB * aabb = [self getAABBFromNode:_node];
	if ( aabb )
	{
		[entity setLocalAABB:aabb];
	}
	
	// set its name
	const char * nameStr = _node->Attribute("id");
	if ( nameStr )
	{
		[entity setName:[NSString stringWithCString:nameStr]];
	}
	else
	{
		[entity setName:@"unknown"];	
	}
	
	// add it to our scene
	if ( _parentEntity != nil )
	{
		[_parentEntity addChild:entity];
	}
	else
	{
		[scene addSceneEntity:entity];
	}
	
	// look for all instance tags and append node to MeshMap
	for ( TiXmlNode * i = _node->FirstChild( COLLADA_INSTANCE_STRUCTURE ); i;
					i = _node->IterateChildren( COLLADA_INSTANCE_STRUCTURE, i ) )
	{
		TiXmlElement * instance = i->ToElement();
		if ( instance )
		{
			// get the url attribute value
			const char * url = instance->Attribute( COLLADA_URL_PROPERTY );
			assert( url && "Could not get the url attribute" );
			
			// remove the #
			++url;
			TiXmlElement * geomNode;
			
			// does this node have a geometry instance?
			geomNode = getGeometryNode( url );
			
			if ( geomNode )
			{
				const char * geomId = geomNode->Attribute( COLLADA_ID_PROPERTY );
				assert( geomId && "Could not get the geomId attribute" );

				// lets add the geometry
				[self addGeometry:geomNode toSceneEntity:entity];
			}
		}
	}
	
	for ( TiXmlNode * i = _node->FirstChild( COLLADA_NODE_STRUCTURE ); i;
					i = _node->IterateChildren( COLLADA_NODE_STRUCTURE, i ) )
	{
		TiXmlElement * node  = i->ToElement();
		if ( !node )
		{
			continue;
		}
		[self processNode:node parentEntity:entity];
	}
	
	[entity release];
	
	// entity now lives in our scene so its OK to return a pointer to it
	return entity;
}

- (void)processNodes:(TiXmlNode *)_scene
{
	int rootEntityCount = 0;	
	rootEntity = nil;
	SBSceneEntity * aRootEntity;
	
	for ( TiXmlNode * i = _scene->FirstChild( COLLADA_NODE_STRUCTURE ); i;
					i = _scene->IterateChildren( COLLADA_NODE_STRUCTURE, i ) )
	{
		TiXmlElement * node  = i->ToElement();
		if (!node)
		{
			continue;
		}
		
		rootEntityCount++;
		aRootEntity = [self processNode:node parentEntity:nil];
	}
	
	if ( rootEntityCount == 1 )
	{
		rootEntity = aRootEntity;
	}
}

- (id)init
{
	if ( self = [super init] )
	{
		globalStreamsCache = [[NSMutableDictionary alloc] init];
		return self;
	}
	return nil;
}

- (void)dealloc
{
	[globalStreamsCache release];
	[super dealloc];
}

- (SBSceneEntity *) loadObjectWithPath:(NSString *)_scenePath
{	
	SBScene * tempScene = [[SBScene alloc] init];
	if ( ![self loadSceneWithPath: [NSString stringWithFormat:@"%@/%@", [[NSBundle mainBundle] resourcePath], _scenePath]
						intoScene: tempScene] )
	{
		NSLog( @"Could not load %@", _scenePath );
		return nil;
	}
	
	SBSceneEntity * obj = [[self rootEntity] retain];
	if ( !obj )
	{
		NSLog( @"Root node not found. Possibly too many root nodes?" );
		return nil;
	}
	
	return [obj autorelease];
}

- (BOOL)loadSceneWithPath:(NSString *)_scenePath intoScene:(SBScene *)_intoScene
{
	if ( !_intoScene )
	{
		return NO;
	}
	scene = _intoScene;
	
	TiXmlDocument doc( [_scenePath UTF8String] );
	if ( !doc.LoadFile() )
	{
		return NO;
	}
	
	root = doc.RootElement();
	if ( !root )
	{
		return NO;
	}
	
	TiXmlNode * xmlScene = root->FirstChild( COLLADA_SCENE_STRUCTURE );
	if ( !xmlScene )
	{
		return 0;
	}
	
	AABB * sceneAABB = [self getAABBFromNode:xmlScene];
	if ( sceneAABB )
	{
		[_intoScene setAABB:sceneAABB];
	}
	
	[self processNodes: xmlScene];
	
	return 1;
}

- (SBSceneEntity *)rootEntity { return rootEntity; }

@end
