//
//  SBGLView.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBGLView.h"
#import <OpenGL/OpenGL.h>
#import <OpenGL/gl.h>
#import <OpenGL/glu.h>

#import "Controller.h"

#import "SBStream.h"
#import "SBDrawable.h"

#import "AABB.h"
#import "SBMat4.h"
#import "SBVector3D.h"
#import "SBVector4D.h"
#import "SBVectorRGB.h"
#import "SBOctreeNode.h"
#import "SBLight.h"

#import "SBScene.h"
#import "SBSceneEntity.h"
#import "SBCamera.h"

#import "SBPathPoints.h"
#import "SBPathPoint.h"

#import "StringTexture.h"

@implementation SBGLView

#pragma mark ---- Initialisation and Control ----

static CFAbsoluteTime gStartTime = 0.0f;

// set app start time
static void setStartTime (void)
{	
	gStartTime = CFAbsoluteTimeGetCurrent ();
}

// return float elpased time in seconds since app start
static CFAbsoluteTime getElapsedTime (void)
{	
	return CFAbsoluteTimeGetCurrent () - gStartTime;
}

// allow this view to be the first responder to events
- (BOOL) acceptsFirstResponder
{
	return YES;
}

// called when the view is initialised
- (id)initWithFrame:(NSRect)frameRect
{
	setStartTime();

	NSOpenGLPixelFormatAttribute pixelFormatAttributes[] =
	{
		NSOpenGLPFAWindow,
		NSOpenGLPFADepthSize,
		NSOpenGLPFASingleRenderer,
		NSOpenGLPFANoRecovery,
		NSOpenGLPFAScreenMask,
		CGDisplayIDToOpenGLDisplayMask( kCGDirectMainDisplay ),
		NSOpenGLPFADoubleBuffer,
		0
	};

	// init the open gl window context
	NSOpenGLPixelFormat * pixelFormat = [[NSOpenGLPixelFormat alloc]
		initWithAttributes: pixelFormatAttributes];
	if ( pixelFormat == nil )
	{
		return nil;
	}
	[pixelFormat autorelease];
	
	self = [super initWithFrame: frameRect pixelFormat:pixelFormat];
	if ( self == nil )
	{
		return nil;
	}
	
	// init the open gl full screen context
	pixelFormatAttributes[0] = NSOpenGLPFAFullScreen;
	NSOpenGLPixelFormat * fullScreenPixelFormat =
		[[NSOpenGLPixelFormat alloc] initWithAttributes: pixelFormatAttributes];
	if ( fullScreenPixelFormat == nil )
	{
		return nil;
	}
	[fullScreenPixelFormat autorelease];
	
	fullScreenGLContext = [[NSOpenGLContext alloc] initWithFormat: fullScreenPixelFormat
		shareContext:[self openGLContext]];
	if ( fullScreenGLContext == nil )
	{
		[self dealloc];
		return nil;
	}
	
	// set up initial variables
	fullScreen = NO;
	wireframe = NO;
	drawInfo = YES;
	drawWorldAABBs = NO;
	drawOctree = NO;
	drawOctreeDepth = -1;
	drawUsingOctree = 1;
	testInFrustum = NO;
	drawPathPoints = NO;
	
	msgPresistance = 10.0f;
	
	// set up the camera
	camera = [[SBCamera alloc] init];
	if ( camera == nil )
	{
		NSLog( @"Error creating camera" );
		return nil;
	}

	// alloc the frustum planes, using Vec4 for A,B,C & D components of a plane
	frustumPlanes = [[NSMutableArray alloc] init];
	int i;
	for ( i = 0; i < 5; ++i )
	{
		[frustumPlanes addObject:[SBVector4D vector4DWithValuesX:0 y:0 z:0 w:0]];
	}
	
	// setup a mutable set and timer for key press handling
	keysPressed = [[NSMutableSet alloc] init];
	[NSTimer scheduledTimerWithTimeInterval:0.03
                                     target:self
                                   selector:@selector(processKeys)
                                   userInfo:nil repeats:YES];
	
	// register for message notifications, ie. things we display as opengl messages
	[[NSNotificationCenter defaultCenter] addObserver:self
											 selector:@selector(handleMessageNotification:)
												 name:@"SBMessage" object:nil];
	
	return self;
}

- (void)dealloc
{
	[fullScreenGLContext release];
	
	[camera release];
	[frustumPlanes release];
	[keysPressed release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	if ( path )
		[path release];
	
	if ( stanStringAttrib )
		[stanStringAttrib release];
	if ( gameStringAttrib )
		[gameStringAttrib release];
	
	if ( infoStringTex )
		[infoStringTex release];
	if ( camStringTex )
		[camStringTex release];
	if ( msgStringTex )
		[msgStringTex release];
	if ( statsStringTex )
		[statsStringTex release];
	
	if ( scoreStringTex )
		[scoreStringTex release];
	if ( livesStringTex )
		[livesStringTex release];
	
	[super dealloc];
}

- (void)toggleFullScreen
{
	if ( fullScreen )
	{
		[fullScreenGLContext clearDrawable];
		[[self openGLContext] makeCurrentContext];
		CGReleaseAllDisplays();
		fullScreen = NO;
	}
	else
	{
		CGCaptureAllDisplays();
		[fullScreenGLContext setFullScreen];
		[fullScreenGLContext makeCurrentContext];
		fullScreen = YES;
	}
}

// wireframe handlers
- (BOOL)wireframe { return wireframe; }
- (void)setWireframe:(BOOL)newWireframe
{
	[self postMessageString:[NSString stringWithFormat:@"Set wireframe:%d", (BOOL)newWireframe]];
	wireframe = newWireframe;
}

// draw info handlers
- (BOOL)drawInfo { return drawInfo; }
- (void)setDrawInfo:(BOOL)newDrawInfo
{
	drawInfo = newDrawInfo;
}

// local aabbs
- (BOOL) drawLocalAABBs { return drawLocalAABBs; }
- (void) setDrawLocalAABBs:(BOOL)newDrawLocalAABBs
{
	[self postMessageString:[NSString stringWithFormat:@"Set draw local AABBs:%d", (BOOL)newDrawLocalAABBs]];
	drawLocalAABBs = newDrawLocalAABBs;
}

// world aabbs
- (BOOL) drawWorldAABBs { return drawWorldAABBs; }
- (void) setDrawWorldAABBs:(BOOL)newDrawWorldAABBs
{
	[self postMessageString:[NSString stringWithFormat:@"Set draw world AABBs:%d", (BOOL)newDrawWorldAABBs]];
	drawWorldAABBs = newDrawWorldAABBs;
}

// octree
- (BOOL) drawOctree { return drawOctree; }
- (void) setDrawOctree:(BOOL)newDrawOctree
{
	[self postMessageString:[NSString stringWithFormat:@"Set draw octree:%d", (BOOL)newDrawOctree]];
	drawOctree = newDrawOctree;
}

// octree depth
- (int) drawOctreeDepth { return drawOctreeDepth; }
- (void) setDrawOctreeDepth:(int)newDrawOctreeDepth
{
	// clamp the values no less than -1 and no greater than the max depth of the octree
	drawOctreeDepth = newDrawOctreeDepth > [SBOctreeNode maxDepth] ? [SBOctreeNode maxDepth] : newDrawOctreeDepth;
	drawOctreeDepth = newDrawOctreeDepth < -1 ? -1 : drawOctreeDepth;
	
	NSString * message;
	if ( newDrawOctreeDepth >= 0 )
	{
		if ( drawOctreeDepth == [SBOctreeNode maxDepth] )
		{
			message = [NSString stringWithFormat:@"Drawing octree at maximum depth:%d", drawOctreeDepth];
		}
		else
		{
			message = [NSString stringWithFormat:@"Drawing octree at depth:%d", drawOctreeDepth];
		}
	}
	else
	{
		message = @"Drawing octree at all depths";
	}
	
	[self postMessageString:message];
}

// draw using the octree
- (BOOL) drawUsingOctree { return drawUsingOctree; }
- (void) setDrawUsingOctree:(BOOL)newDrawUsingOctree
{
	[self postMessageString:[NSString stringWithFormat:@"Set drawing using octree:%d", (BOOL)newDrawUsingOctree]];
	drawUsingOctree = newDrawUsingOctree;
}

// test if an octree node is in the frustum
- (BOOL) testInFrustum { return testInFrustum; }
- (void) setTestInFrustum:(BOOL)newTestInFrustum
{
	[self postMessageString:[NSString stringWithFormat:@"Set test octree in frustum:%d", (BOOL)newTestInFrustum]];
	testInFrustum = newTestInFrustum;
}

// draw the AI path points
- (BOOL) drawPathPoints { return drawPathPoints; }
- (void) setDrawPathPoints:(BOOL)newDrawPathPoints
{
	[self postMessageString:[NSString stringWithFormat:@"Set draw path points:%d", (BOOL)newDrawPathPoints]];
	drawPathPoints = newDrawPathPoints;
}

- (SBCamera *)camera
{
	return camera;
}

- (void) setPath:(NSArray *)_path
{
	path = [_path retain];
}

#pragma mark ---- Drawing ----

- (void)prepareOpenGL
{
	glEnable( GL_TEXTURE_2D );                // Enable texture mapping
	glShadeModel( GL_SMOOTH );                // Enable smooth shading
	glEnable(GL_CULL_FACE);
	glCullFace( GL_BACK );
	glClearColor( 0.0f, 0.0f, 0.0f, 0.5f );   // Black background
	glClearDepth( 1.0f );                     // Depth buffer setup
	glEnable( GL_DEPTH_TEST );                // Enable depth testing
	glDepthFunc( GL_LEQUAL );                 // Type of depth test to do
	glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );
	
	GLfloat material_specular[] = { 1.0, 1.0, 1.0, 1,0 };
	GLfloat material_emission[] = { 0.0, 0.0, 0.0, 1.0 };
	
	glEnable( GL_LIGHTING );
	glEnable( GL_COLOR_MATERIAL );
	glColorMaterial( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE );
	glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR, material_specular );
	glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, material_emission );
		
	[camera resetCamera];	
	
	// init fonts for use with strings
	NSFont * font = [NSFont fontWithName:@"Helvetica" size:12.0];
	stanStringAttrib = [[NSMutableDictionary dictionary] retain];
	[stanStringAttrib setObject:font forKey:NSFontAttributeName];
	[stanStringAttrib setObject:[NSColor whiteColor] forKey:NSForegroundColorAttributeName];
	[font release];
	
	font = [NSFont fontWithName:@"Comic Sans MS" size:24];
	if ( !font )
	{
		font = [NSFont fontWithName:@"Helvetica" size:12.0];
	}
	gameStringAttrib = [[NSMutableDictionary dictionary] retain];
	[gameStringAttrib setObject:font forKey:NSFontAttributeName];
	[gameStringAttrib setObject:[NSColor whiteColor] forKey:NSForegroundColorAttributeName];
	[font release];
	
	// ensure strings are created
	[self createMessageString];
	[self updateStatsString];
	
	// set mouse moved responder on window
//	NSWindow * viewWindow = [self window];
//	if ( !viewWindow )
//	{
//		NSLog( @"No window found for view" );
//		return;
//	}
//	[viewWindow setAcceptsMouseMovedEvents:true];
}

- (void)drawAABB:(AABB *)aabb withColorR:(float)r g:(float)g b:(float)b
{
	// temporarily disable backface culling
	glDisable(GL_CULL_FACE);
	
	glColor3f(r,g,b);
	
	glPolygonMode(GL_FRONT, GL_LINE );
    glPolygonMode(GL_BACK, GL_LINE );
	
	SBVector3D * minVec = [aabb min], * maxVec = [aabb max];
	float min[3] = { [minVec x], [minVec y], [minVec z] };
	float max[3] = { [maxVec x], [maxVec y], [maxVec z] };
	
	glBegin( GL_POLYGON );
		// front face - facing pos z
		glVertex3f (max[0], max[1], max[2]);
		glVertex3f (min[0], max[1], max[2]);
		glVertex3f (min[0], min[1], max[2]);
		glVertex3f (max[0], min[1], max[2]);
	glEnd ();
	
	glBegin( GL_POLYGON );
		// back face
		glVertex3f (min[0], min[1], min[2]);
		glVertex3f (max[0], min[1], min[2]);
		glVertex3f (max[0], max[1], min[2]);
		glVertex3f (min[0], max[1], min[2]);
	glEnd ();
	
	glBegin( GL_POLYGON );
		// left face
		glVertex3f (min[0], min[1], min[2]);
		glVertex3f (min[0], min[1], max[2]);
		glVertex3f (min[0], max[1], max[2]);
		glVertex3f (min[0], max[1], min[2]);
	glEnd ();
	
	glBegin( GL_POLYGON );
		// right face
		glVertex3f (max[0], max[1], max[2]);
		glVertex3f (max[0], min[1], max[2]);
		glVertex3f (max[0], min[1], min[2]);
		glVertex3f (max[0], max[1], min[2]);
	glEnd ();
	
	glBegin( GL_POLYGON );
		// top face
		glVertex3f (max[0], max[1], max[2]);
		glVertex3f (max[0], max[1], min[2]);
		glVertex3f (min[0], max[1], min[2]);
		glVertex3f (min[0], max[1], max[2]);
	glEnd ();
	
	glBegin( GL_POLYGON );
		// bot face
		glVertex3f (min[0], min[1], min[2]);
		glVertex3f (max[0], min[1], min[2]);
		glVertex3f (max[0], min[1], max[2]);
		glVertex3f (min[0], min[1], max[2]);
	glEnd ();
	
	glEnable(GL_CULL_FACE);
}

- (void)drawWorldAABBs:(SBSceneEntity *)entity
{	
	[self drawAABB:[entity worldAABB] withColorR:1 g:0 b:0];
	
	// draw the children
	NSEnumerator * childEn = [[entity children] objectEnumerator];
	SBSceneEntity * child;
	while ( child = [childEn nextObject] )
	{
		[self drawWorldAABBs:child];
	}
}

- (void)drawAllPathPoints;
{
	SBPathPoints * pathPoints = [[Controller scene] pathPoints];
	NSEnumerator * en = [[pathPoints points] objectEnumerator];
	SBPathPoint * point;
	
	glColor3f( 1, 0, 0 );
	glLineWidth( 2 );
	
	GLUquadric * quad = gluNewQuadric();
	gluQuadricOrientation( quad,GLU_OUTSIDE );
	
	float * mulMat = malloc( sizeof(float)*16 );
	
	while ( point = [en nextObject] )
	{
		SBVector3D * t = [point translation];
		SBMat4 * mat4 = [SBMat4 mat4WithTranslation:t];
		[mat4 putContiguousDataInto:mulMat];

		glPushMatrix();
		glMultMatrixf( mulMat );

		gluSphere( quad, 0.05, 10, 10 );
		
		glPopMatrix();

		NSEnumerator * neighbours = [[point neighbours] objectEnumerator];
		SBPathPoint * neighbour;
		while ( neighbour = [neighbours nextObject] )
		{
			glBegin( GL_LINES );
			glVertex3f( [t x], [t y], [t z] );
			SBVector3D * nTrans = [neighbour translation];
			glVertex3f( [nTrans x], [nTrans y], [nTrans z] );
			glEnd();
		}
	}
	
	free( mulMat );
}

- (void)drawPath:(NSArray *)_path
{	
	glColor3f( 0, 1, 1 );
	glLineWidth( 3 );
	
	unsigned int i;
	for ( i = 0; i < [_path count] - 1; ++i )
	{
		SBVector3D * p1 = [[_path objectAtIndex:i] translation];
		SBVector3D * p2 = [[_path objectAtIndex:i+1] translation];
		
		glBegin( GL_LINES );
		glVertex3f( [p1 x], [p1 y], [p1 z] );
		glVertex3f( [p2 x], [p2 y], [p2 z] );
		glEnd();
	}
}

- (void)drawOctree:(SBOctreeNode *)octNode
{
	if ( drawOctreeDepth < 0 || [octNode depth] == drawOctreeDepth )
	{
		[self drawAABB:[octNode extents] withColorR:0 g:1 b:0];

		if ( drawOctreeDepth >= 0 )
		{
			// don't need to draw any children
			return;
		}
	}
	
	// draw the children
	NSEnumerator * childEn = [[octNode children] objectEnumerator];
	SBOctreeNode * child;
	while ( child = [childEn nextObject] )
	{
		[self drawOctree:child];
	}
}

- (void)drawOctreeContents:(SBOctreeNode *)octNode frustumTest:(BOOL)nodeNeedsFrustumTest
{
	BOOL childrenNodesNeedTest = YES; // default to yes in case we aren't testing frustum
	
	// if we are drawing all octreeNodes or we are drawing at this nodes depth
	if ( drawOctreeDepth < 0 || [octNode depth] == drawOctreeDepth )
	{
		// if the global option is to test the frustum, and our caller knows we should test this node
		if ( testInFrustum && nodeNeedsFrustumTest )
		{
			// test if this node is in the frustum and therefore we should draw it
			int frustumTest = [self aabbInFrustum:[octNode extents]];
			if ( frustumTest == 0 )
			{
				return;
			}
			else if ( frustumTest == 2 )
			{
				// we know all children need to be rendered so don't test frustum
				childrenNodesNeedTest = NO;
			}
		}
		
		// draw the entities in this octNode
		NSEnumerator * entEn = [[octNode objects] objectEnumerator];
		SBSceneEntity * entity;
		while ( entity = [entEn nextObject] )
		{
			[self drawEntity:entity inLocalSpace:NO];
		}

		if ( drawOctreeDepth >= 0 )
		{
			// don't need to draw any children as we are only drawing at this depth
			return;
		}
	}
	
	NSEnumerator * octEn = [[octNode children] objectEnumerator];
	SBOctreeNode * child;
	while ( child = [octEn nextObject] )
	{
		[self drawOctreeContents:child frustumTest:childrenNodesNeedTest];
	}
}

- (void)drawEntity:(SBSceneEntity *)entity inLocalSpace:(BOOL)localSpace
{	
	numDrawnObjects++;
	
	float * mulMat = malloc( sizeof(float)*16 );
	glPushMatrix();

	if ( localSpace )
	{
		// push the scene entity matrix
		[[entity localTransform] putContiguousDataInto:mulMat];
	}
	else
	{
		[[entity worldTransform] putContiguousDataInto:mulMat];		
	}

	glMultMatrixf( mulMat );
	free( mulMat );
	
	if ( drawLocalAABBs )
	{
		[self drawAABB:[entity localAABB] withColorR:0 g:0 b:1];
	}
	else
	{
		// draw the entity
		NSEnumerator * drawEn = [[entity drawables] objectEnumerator];
		SBDrawable * drawable;
		BOOL useAltTexIdx = [entity useAlternateTexture];
		
		while ( drawable = [ drawEn nextObject ] )
		{			
			SBVertexStream * position = [drawable streamWithType:SI_Position];
			SBVertexStream * normal = [drawable streamWithType:SI_Normal];
			SBVertexStream * uv0 = [drawable streamWithType:SI_UV0];
			
			if ( position != nil )
			{
				glVertexPointer(3, GL_FLOAT, 0, [position data] );
				glEnableClientState( GL_VERTEX_ARRAY );
			}
			if ( normal != nil )
			{
				glNormalPointer(GL_FLOAT, 0, [normal data] );
				glEnableClientState( GL_NORMAL_ARRAY );
			}
			if ( uv0 != nil )
			{
				glTexCoordPointer( 2, GL_FLOAT, 0, [uv0 data] );
				glEnableClientState( GL_TEXTURE_COORD_ARRAY );
			}
			
			GLuint textureIdx;
			if ( useAltTexIdx )
			{
				textureIdx = [drawable alternateTextureIdx];
			}
			else
			{
				textureIdx = [drawable textureIdx];
			}
				
			if ( textureIdx >= 0 )
			{
				glBindTexture( GL_TEXTURE_2D, textureIdx );
			}
			
			glDrawElements( GL_TRIANGLES, [drawable numIndices], GL_UNSIGNED_SHORT,
							[[drawable indexStream] data] );
			
			glDisableClientState( GL_VERTEX_ARRAY );
			glDisableClientState( GL_NORMAL_ARRAY );
			glDisableClientState( GL_TEXTURE_COORD_ARRAY );
		}
	}
	
	if ( localSpace )
	{
		// draw the children
		NSEnumerator * childEn = [[entity children] objectEnumerator];
		SBSceneEntity * child;
		while ( child = [childEn nextObject] )
		{
			[self drawEntity:child inLocalSpace:YES];
		}
	}
	
	// pop the matrix
	glPopMatrix();
}

- (void)drawRect:(NSRect)rect
{
	[self reshape]; // forces projection matrix update (does test for size changes)
	[self updateModelView];  // update model view matrix for object
	[self calculateFrustum];
	
	// Clear the screen and depth buffer
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	glColor3f( 1.0f, 1.0f, 1.0f );
	glPolygonMode(GL_FRONT, wireframe ? GL_LINE : GL_FILL);
    glPolygonMode(GL_BACK, wireframe ? GL_LINE : GL_FILL);
	
	SBScene * scene = [Controller scene];
	
	numDrawnObjects = 0;
	
	if ( scene != nil && [[scene sceneEntities] count] > 0 )
	{
		// don't enable lighting when debugging AABBs
		if ( !drawLocalAABBs && !drawWorldAABBs )
		{
			glEnable( GL_LIGHTING );
			
			// set up all the lights
			NSEnumerator * en = [[scene lights] objectEnumerator];
			SBLight * light;
			while ( light = [en nextObject] )
			{
				SBVector3D * pos = [light position];
				GLfloat posArray[] = { [pos x], [pos y], [pos z], 0.0 };
				
				glEnable( [light lightNum] );
				glLightfv( [light lightNum], GL_POSITION, posArray );
				
				float * diff = [[light diffuse] asFloat4];
				
				glLightfv( [light lightNum], GL_AMBIENT, [[light ambient] asFloat4] );
				glLightfv( [light lightNum], GL_DIFFUSE, diff );
				glLightfv( [light lightNum], GL_SPECULAR, [[light specular] asFloat4] );
				glLightfv( [light lightNum], GL_POSITION, posArray );
			}
		}
		
		SBSceneEntity * entity;
		if ( drawWorldAABBs )
		{
			NSEnumerator * sceneEn2 = [[scene sceneEntities] objectEnumerator];
			while ( entity = [sceneEn2 nextObject] )
			{
				[self drawWorldAABBs:entity];
			} 
		}
		else
		{
			
			if ( drawUsingOctree )
			{
				[self drawOctreeContents:[scene rootOctreeNode] frustumTest:YES];
			}
			else
			{
				NSEnumerator * sceneEn = [[scene sceneEntities] objectEnumerator];
				while ( entity = [sceneEn nextObject] )
				{
					[self drawEntity:entity inLocalSpace:YES];
				}
			}
		}
		
		glDisable( GL_LIGHTING );
		
		if ( drawOctree )
		{
			[self drawOctree:[scene rootOctreeNode]];
		}
		
		if ( drawPathPoints )
		{
			[self drawAllPathPoints];
			if ( path )
			{
				[self drawPath:path];
			}
		}
	}
	
	[self updateStatsString];
	[self updateScoreString];
	[self updateLivesString];
	[self drawInfoInView];
	
	[[self openGLContext] flushBuffer];
}

//- (void) drawRemainingLives
//{
//	NSString * bundlePath = [[NSBundle mainBundle] resourcePath];
//	NSString * texturePath = [NSString stringWithFormat:@"%@/%s", bundlePath, "pac.tga"];
//	
//	[self drawTexture:texturePath atPoint:NSMakePoint(80,80)];
//}
//
//- (void) drawTexture:(NSString *)texturePath atPoint:(NSPoint)point
//{
//	NSDictionary * texEntry = [Controller lookupGLTexture:texturePath];
//	
//	if ( texEntry )
//	{
//		// determine the texture width and height
//		[self drawTexture:texEntry withBounds:NSMakeRect (point.x, point.y, 150, 150)];
//	}
//}
//
//- (void) drawTexture:(NSDictionary *)texEntry withBounds:(NSRect)bounds
//{
//	float width = [[texEntry objectForKey:@"width"] floatValue];
//	float height = [[texEntry objectForKey:@"height"] floatValue];
//	GLuint texId = [[texEntry objectForKey:@"id"] unsignedIntValue];
//	
//	glBindTexture( GL_TEXTURE_2D, texId );
//	
//	glBegin( GL_QUADS );
//	glTexCoord2f( 0.0f, 0.0f ); // draw upper left in world coordinates
//	glVertex2f( bounds.origin.x, bounds.origin.y );
//	
//	glTexCoord2f( 0.0f, height ); // draw lower left in world coordinates
//	glVertex2f( bounds.origin.x, bounds.origin.y + bounds.size.height );
//	
//	glTexCoord2f( width, height ); // draw upper right in world coordinates
//	glVertex2f( bounds.origin.x + bounds.size.width, bounds.origin.y + bounds.size.height );
//	
//	glTexCoord2f( width, 0.0f ); // draw lower right in world coordinates
//	glVertex2f( bounds.origin.x + bounds.size.width, bounds.origin.y );
//	glEnd();	
//}

// returns 0 for no points, 1 for some, and 2 for all points in frustum
- (int)aabbInFrustum:(AABB *)aabb
{
	SBVector3D * min = [aabb min];
	SBVector3D * max = [aabb max];
	
	// number of points and planes satisfied
	int points, planes = 0;
	
	NSEnumerator * en = [frustumPlanes objectEnumerator];
	SBVector4D * p;
	while ( p = [en nextObject] )
	{
		points = 0;
		// min, min, min
		if ( [p x] * [min x] + [p y] * [min y] + [p z] * [min z] + [p w] > 0 )
			points++;
		// max, min, min
		if ( [p x] * [max x] + [p y] * [min y] + [p z] * [min z] + [p w] > 0 )
			points++;
		// min, max, min
		if ( [p x] * [min x] + [p y] * [max y] + [p z] * [min z] + [p w] > 0 )
			points++;
		// max, max, min
		if ( [p x] * [max x] + [p y] * [max y] + [p z] * [min z] + [p w] > 0 )
			points++;
		// min, min, max
		if ( [p x] * [min x] + [p y] * [min y] + [p z] * [max z] + [p w] > 0 )
			points++;
		// max, min, max
		if ( [p x] * [max x] + [p y] * [min y] + [p z] * [max z] + [p w] > 0 )
			points++;
		// min, max, max
		if ( [p x] * [min x] + [p y] * [max y] + [p z] * [max z] + [p w] > 0 )
			points++;
		// max, max, max
		if ( [p x] * [max x] + [p y] * [max y] + [p z] * [max z] + [p w] > 0 )
			points++;
		
		// if there are no points in the frustum then we can bail here
		if ( points == 0 )
			return NO;
		
		// if all points are satisfied for this plane then increment num planes satsified
		if ( points == 8 )
			planes++;
	}
	
	return (planes == 6 ) ? 2 : 1;
}

- (void)calculateFrustum
{
	float proj[16];
	float modl[16];

	glGetFloatv( GL_PROJECTION_MATRIX, proj );
	SBMat4 * projM = [[SBMat4 alloc] initWithFloats:proj];

	glGetFloatv( GL_MODELVIEW_MATRIX, modl );
	SBMat4 * modlM = [[SBMat4 alloc] initWithFloats:modl];
	
	SBMat4 * clipM = [[projM multiplyByMat4:modlM] retain];

	SBVector4D * rightPlane = [frustumPlanes objectAtIndex:0];
	[rightPlane setWithValuesX:[[clipM col0] w] - [[clipM col0] x]
							 y:[[clipM col1] w] - [[clipM col1] x]
							 z:[[clipM col2] w] - [[clipM col2] x]
							 w:[[clipM col3] w] - [[clipM col3] x]];
	
	SBVector4D * leftPlane = [frustumPlanes objectAtIndex:1];
	[leftPlane setWithValuesX:[[clipM col0] w] + [[clipM col0] x]
							y:[[clipM col1] w] + [[clipM col1] x]
							z:[[clipM col2] w] + [[clipM col2] x]
							w:[[clipM col3] w] + [[clipM col3] x]];

	SBVector4D * topPlane = [frustumPlanes objectAtIndex:2];
	[topPlane setWithValuesX:[[clipM col0] w] - [[clipM col0] y]
							  y:[[clipM col1] w] - [[clipM col1] y]
							  z:[[clipM col2] w] - [[clipM col2] y]
							  w:[[clipM col3] w] - [[clipM col3] y]];

	SBVector4D * bottomPlane = [frustumPlanes objectAtIndex:3];
	[bottomPlane setWithValuesX:[[clipM col0] w] + [[clipM col0] y]
							  y:[[clipM col1] w] + [[clipM col1] y]
							  z:[[clipM col2] w] + [[clipM col2] y]
							  w:[[clipM col3] w] + [[clipM col3] y]];

	SBVector4D * farPlane = [frustumPlanes objectAtIndex:4];
	[farPlane setWithValuesX:[[clipM col0] w] - [[clipM col0] z]
						   y:[[clipM col1] w] - [[clipM col1] z]
						   z:[[clipM col2] w] - [[clipM col2] z]
						   w:[[clipM col3] w] - [[clipM col3] z]];

	SBVector4D * nearPlane = [frustumPlanes objectAtIndex:4];
	[nearPlane setWithValuesX:[[clipM col0] w] + [[clipM col0] z]
							y:[[clipM col1] w] + [[clipM col1] z]
							z:[[clipM col2] w] + [[clipM col2] z]
							w:[[clipM col3] w] + [[clipM col3] z]];
}

- (void)reshape
{
	NSRect rectView = [self bounds];
	
	// ensure camera knows size changed
	if (( [camera viewHeight] != rectView.size.height ) || ( [camera viewWidth] != rectView.size.width ))
	{
		[camera setHeight:rectView.size.height];
		[camera setWidth:rectView.size.width];
		
		glViewport ( 0, 0, [camera viewWidth], [camera viewHeight] );
		[self updateProjection];  // update projection matrix
	}
}

// this can be a troublesome call to do anything heavyweight, as it is called on window moves,
//  - resizes, and display config changes.  So be careful of doing too much here.
- (void) update
{
	[self postMessageString:[NSString stringWithFormat:@"update at %0.1f secs", msgTime]];

	[super update];
	
	if (![self inLiveResize])
	{ // if not doing live resize
		[self updateInfoString];
	}
}

// update the projection matrix based on camera and view info
- (void) updateProjection
{
	GLdouble ratio, radians, wd2;
	GLdouble left, right, top, bottom, near, far;

    [[self openGLContext] makeCurrentContext];

	// set projection
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity ();
	SBVector3D * viewPos = [camera viewPos];
	near = -[viewPos z] - 40;
	if (near < 0.05)
		near = 0.05;
	far = -[viewPos z] + 40;
	radians = 0.0174532925 * [camera aperture] / 2; // half aperture degrees to radians 
	wd2 = near * tan(radians);
	ratio = [camera viewWidth] / (float) [camera viewHeight];
	if (ratio >= 1.0) {
		left  = -ratio * wd2;
		right = ratio * wd2;
		top = wd2;
		bottom = -wd2;	
	} else {
		left  = -wd2;
		right = wd2;
		top = wd2 / ratio;
		bottom = -wd2 / ratio;	
	}
	glFrustum (left, right, bottom, top, near, far);
	
	[self updateCameraString];
}

// updates the contexts model view matrix for object and camera moves
- (void) updateModelView
{
    [[self openGLContext] makeCurrentContext];
	
	// move view
	glMatrixMode (GL_MODELVIEW);
	glLoadIdentity ();
	
	SBVector3D * viewPos = [camera viewPos];
	SBVector3D * viewDir = [camera viewDir];
	SBVector3D * viewUp = [camera viewUp];
	
	gluLookAt ([viewPos x], [viewPos y], [viewPos z],
			  [viewPos x] + [viewDir x], [viewPos y] + [viewDir y], [viewPos z] + [viewDir z],
			  [viewUp x], [viewUp y], [viewUp z]);
		
	[self updateCameraString];
}

#pragma mark ---- Event Handling ----

//- (void)mouseMoved:(NSEvent *)theEvent
//{
//	NSPoint currMouseLoc = [self convertPoint:[theEvent locationInWindow] fromView:nil];
//	NSRect bounds = [self bounds];
//	
//	// normalise the point
//	currMouseLoc.x = currMouseLoc.x / bounds.size.width;
//	currMouseLoc.y = currMouseLoc.y / bounds.size.height;
//	
//	float sensitivity = 20; // higher value, faster mouse
//	float xDist = (currMouseLoc.x - prevMouseLoc.x) * sensitivity;
//	//float yDist = (currMouseLoc.y - prevMouseLoc.y) * sensitivity;
//	
//	prevMouseLoc = currMouseLoc;
//	
//	[camera turnWithRotation:xDist];
//}
	
- (void)keyDown:(NSEvent *)theEvent
{
	NSNumber * key = [NSNumber numberWithUnsignedInt:[[theEvent characters] characterAtIndex:0]];
	
	switch ( [key unsignedIntValue] )
	{		
		// increase and decrease the octree draw depth
		case NSUpArrowFunctionKey:
			[self setDrawOctreeDepth:drawOctreeDepth+1];
			break;
		case NSDownArrowFunctionKey:
			[self setDrawOctreeDepth:drawOctreeDepth-1];
			break;
			
		case '8' :
		case '5' :
		case '4' :
		case '6' :
		case 'z' : case 'Z' :
		case 'x' : case 'X' :
		case 'd' : case 'D' :
		case 'c' : case 'C' :
			[keysPressed addObject:key];
			break;
		default:
			break;
	}
}

- (void)keyUp:(NSEvent *)theEvent
{
	NSNumber * key = [NSNumber numberWithUnsignedInt:[[theEvent characters] characterAtIndex:0]];
	
	switch ( [key unsignedIntValue] )
	{			
		case '8' :
		case '5' :
		case '4' :
		case '6' :
		case 'z' : case 'Z' :
		case 'x' : case 'X' :
		case 'd' : case 'D' :
		case 'c' : case 'C' :
			[keysPressed removeObject:key];
			break;
		default:
			break;
	}
}

- (void)processKeys
{
	if ( [keysPressed count] > 0 )
	{
		NSEnumerator * en = [keysPressed objectEnumerator];
		NSNumber * key;
		while ( key = [en nextObject] )
		{
			switch ( [key unsignedIntValue] )
			{
				case '8' :
					[camera move:1];
					break;
				case '5' :
					[camera move:0];
					break;
				case '4' :
					[camera turn:1];
					break;
				case '6' :
					[camera turn:0];
					break;
				case 'z' : case 'Z':
					[camera strafe:1];
					break;
				case 'x' : case 'X':
					[camera strafe:0];
					break;
				case 'd' : case 'D':
					[camera elevate:1];
					break;
				case 'c' : case 'C':
					[camera elevate:0];
					break;
			}
		}
	}
}

- (void)scrollWheel:(NSEvent *)theEvent
{
	float wheelDelta = [theEvent deltaX] +[theEvent deltaY] + [theEvent deltaZ];
	if (wheelDelta)
	{
		GLfloat aperture = [camera aperture];
		GLfloat deltaAperture = wheelDelta * -aperture / 200.0f;
		aperture += deltaAperture;
		if (aperture < 0.1) // do not let aperture <= 0.1
			aperture = 0.1;
		if (aperture > 179.9) // do not let aperture >= 180
			aperture = 179.9;
		[camera setAperture:aperture];
		[self updateProjection]; // update projection matrix
		[self setNeedsDisplay: YES];
	}
}

#pragma mark ---- Debug Drawing ----

- (void) createMessageString
{
	NSString * string = [NSString stringWithFormat:@"No messages..."];
	msgStringTex = [[StringTexture alloc] initWithString:string withAttributes:stanStringAttrib
				withTextColor:[NSColor colorWithDeviceRed:1.0f green:1.0f blue:1.0f alpha:1.0f]
				withBoxColor:[NSColor colorWithDeviceRed:0.5f green:0.5f blue:0.5f alpha:0.5f]
				withBorderColor:[NSColor colorWithDeviceRed:0.8f green:0.8f blue:0.8f alpha:0.8f]];
}

- (void) handleMessageNotification:(NSNotification *)notification
{
	NSDictionary * dict = [notification userInfo];
	[self postMessageString:[dict objectForKey:@"message"]];
}

- (void) postMessageString: (NSString *) newMessage
{
	msgTime	= getElapsedTime ();
	[msgStringTex setString:newMessage withAttributes:stanStringAttrib];
}

- (void) updateInfoString
{ // update info string texture
	NSString * string = [NSString stringWithFormat:@"(%0.0f x %0.0f) \n%s \n%s",
		[self bounds].size.width, [self bounds].size.height, glGetString (GL_RENDERER), glGetString (GL_VERSION)];
	if (infoStringTex)
		[infoStringTex setString:string withAttributes:stanStringAttrib];
	else {
		infoStringTex = [[StringTexture alloc] initWithString:string withAttributes:stanStringAttrib
				withTextColor:[NSColor colorWithDeviceRed:1.0f green:1.0f blue:1.0f alpha:1.0f]
				withBoxColor:[NSColor colorWithDeviceRed:0.5f green:0.5f blue:0.5f alpha:0.5f]
				withBorderColor:[NSColor colorWithDeviceRed:0.8f green:0.8f blue:0.8f alpha:0.8f]];
	}
}

- (void) updateCameraString
{ // update info string texture
	NSString * string = [NSString stringWithFormat:@"Camera at (%0.1f, %0.1f, %0.1f) looking at (%0.1f, %0.1f, %0.1f) with %0.1f aperture",
			[[camera viewPos] x], [[camera viewPos] y], [[camera viewPos] z],
			[[camera viewDir] x], [[camera viewDir] y], [[camera viewDir] z], [camera aperture]];
	
	if (camStringTex)
		[camStringTex setString:string withAttributes:stanStringAttrib];
	else {
		camStringTex = [[StringTexture alloc] initWithString:string withAttributes:stanStringAttrib
				withTextColor:[NSColor colorWithDeviceRed:1.0f green:1.0f blue:1.0f alpha:1.0f]
				withBoxColor:[NSColor colorWithDeviceRed:0.5f green:0.5f blue:0.5f alpha:0.5f]
				withBorderColor:[NSColor colorWithDeviceRed:0.8f green:0.8f blue:0.8f alpha:0.8f]];
	}
}

- (void) updateStatsString
{
	NSString * string = [NSString stringWithFormat:@"Object Draw Count: %d", numDrawnObjects];
	
	if (statsStringTex)
		[statsStringTex setString:string withAttributes:stanStringAttrib];
	else {
		statsStringTex = [[StringTexture alloc] initWithString:string withAttributes:stanStringAttrib
				   withTextColor:[NSColor colorWithDeviceRed:1.0f green:1.0f blue:1.0f alpha:1.0f]
					withBoxColor:[NSColor colorWithDeviceRed:0.5f green:0.5f blue:0.5f alpha:0.5f]
				 withBorderColor:[NSColor colorWithDeviceRed:0.8f green:0.8f blue:0.8f alpha:0.8f]];
	}
}

- (void) updateScoreString
{
	NSString * string = [NSString stringWithFormat:@"Score: %d", [[Controller scene] score]];
	
	if ( scoreStringTex )
		[scoreStringTex setString:string withAttributes:gameStringAttrib];
	else {
		scoreStringTex = [[StringTexture alloc] initWithString:string withAttributes:gameStringAttrib
												 withTextColor:[NSColor colorWithDeviceRed:0.9f green:0.7f blue:1.0f alpha:1.0f]
												  withBoxColor:[NSColor colorWithDeviceRed:0.0f green:0.0f blue:0.0f alpha:0.0f]
											   withBorderColor:[NSColor colorWithDeviceRed:0.0f green:0.0f blue:0.0f alpha:0.0f]];
	}
}

- (void) updateLivesString
{
	NSString * string = [NSString stringWithFormat:@"Lives: %d", [[Controller scene] remainingLives]];
	
	if ( livesStringTex )
		[livesStringTex setString:string withAttributes:gameStringAttrib];
	else {
		livesStringTex = [[StringTexture alloc] initWithString:string withAttributes:gameStringAttrib
												 withTextColor:[NSColor colorWithDeviceRed:0.9f green:0.7f blue:1.0f alpha:1.0f]
												  withBoxColor:[NSColor colorWithDeviceRed:0.0f green:0.0f blue:0.0f alpha:0.0f]
											   withBorderColor:[NSColor colorWithDeviceRed:0.0f green:0.0f blue:0.0f alpha:0.0f]];
	}
}

// draw text info using our StringTexture class for much more optimized text drawing
- (void) drawInfoInView
{	
	glPolygonMode(GL_FRONT, GL_FILL);
	glPolygonMode(GL_BACK, GL_FILL);
	
	GLint matrixMode;
	GLboolean depthTest = glIsEnabled (GL_DEPTH_TEST);
	GLfloat height, width, messageTop = 10.0f;
	
	height = [camera viewHeight];
	width = [camera viewWidth];
	
	glDisable (GL_DEPTH_TEST); // ensure text is not removed by depth buffer
	glEnable (GL_BLEND); // for text fading
	glBlendFunc (GL_ONE, GL_ONE_MINUS_SRC_ALPHA); // ditto
	glEnable (GL_TEXTURE_RECTANGLE_EXT);
	
	// set orthograhic 1:1  pixel transform in local view coords
	glGetIntegerv (GL_MATRIX_MODE, &matrixMode);
	glMatrixMode (GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity ();
	glMatrixMode (GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity ();
	glScalef (2.0f / width, -2.0f /  height, 1.0f);
	glTranslatef (-width / 2.0f, -height / 2.0f, 0.0f);
	
	glColor4f (1.0f, 1.0f, 1.0f, 1.0f);
	
//	if ( [Controller scene] )
//	{
//		[self drawRemainingLives];
//	}
	
	if ( drawInfo )
	{
		[infoStringTex drawAtPoint:NSMakePoint (10.0f, height - [infoStringTex frameSize].height - 10.0f)];
		
		[camStringTex drawAtPoint:NSMakePoint (10.0f, messageTop)];
		messageTop += [camStringTex frameSize].height + 3.0f;
		
		[statsStringTex drawAtPoint:NSMakePoint (10.0f, messageTop)];
		messageTop += [statsStringTex frameSize].height + 3.0f;
		
		// message string
		float currTime = getElapsedTime ();
		if ((currTime - msgTime) < msgPresistance)
		{
			GLfloat comp = (msgPresistance - getElapsedTime () + msgTime) * 0.1; // premultiplied fade
			glColor4f (comp, comp, comp, comp);
			[msgStringTex drawAtPoint:NSMakePoint (10.0f, messageTop)];
			messageTop += [msgStringTex frameSize].height + 3.0f;
		}
	}
	
	[scoreStringTex drawAtPoint:NSMakePoint( width - [scoreStringTex frameSize].width - 10.0f, 10.0f )];
	[livesStringTex drawAtPoint:NSMakePoint( width - [livesStringTex frameSize].width - 10.0f, [scoreStringTex frameSize].height + 3.0f )];
	
	// reset orginal martices
	glPopMatrix(); // GL_MODELVIEW
	glMatrixMode (GL_PROJECTION);
	glPopMatrix();
	glMatrixMode (matrixMode);
	
	glDisable (GL_TEXTURE_RECTANGLE_EXT);
	glDisable (GL_BLEND);
	if (depthTest)
		glEnable (GL_DEPTH_TEST);
}

@end