//
//  SBMaze.m
//  SBGame
//
//  Created by Stuart Bryson on 21/09/05.
//  Copyright 2005 __MyCompanyName__. All rights reserved.
//

#import "SBMaze.h"
#import "SBCell.h"
#import "SBVector3D.h"
#import "SBMat4.h"
#import "SBColladaLoader.h"
#import "SBSceneEntity.h"
#import "SBScene.h"
#import "SBCamera.h"
#import "SBNonuniformSpline.h"
#import "SBEnemy.h"

@implementation SBMaze

- (id) initWithSize:(int)_size
{
	if ( self = [super init] )
	{
		cells = [[NSMutableArray alloc] init];
		
		int x,y;
		size = _size > 1 ? _size : 2; // lets ensure we have at least 2 cells
		for ( x = 0; x < size; ++x )
		{
			for ( y = 0; y < size; ++y )
			{
				[cells addObject:[SBCell cellWithX:x y:y]];
			}
		}

		return self;
	}
	
	return nil;
}

- (void) generateMazeWithCameraSpline:(SBNonuniformSpline *)camSpline
{
	NSMutableArray * stack = [[NSMutableArray alloc] init];
	
	startCell = [self getCellAtX:(rand() % size) y:(rand() % size)];
	SBCell * currCell = startCell;
		
	int visitedCells = 1;
	while ( visitedCells < size*size )
	{
		//NSLog( @"Working cell: (%d,%d)", [currCell x], [currCell y] );

		// find all neightbours of currCell with all walls intact
		NSArray * neighbours = [self getIntactNeighboursOfCell:currCell];
		
		if ( [neighbours count] == 0 )
		{
			//NSLog( @"Found no neighbours. Popping stack" );
			currCell = [stack lastObject];
			[stack removeLastObject];
		}
		else
		{
			// choose one of our neighbouring cells at random
			int randomNeighbour = rand() % [neighbours count];
					
			// knock down the wall between our currCell and randomNeighbour
			SBCell * nextCell = [neighbours objectAtIndex:randomNeighbour];
			
			// we use secondCell to point the camera in the correct direction
			if ( !secondCell )
			{
				secondCell = nextCell;
			}
			
			//NSLog( @"Chose neighbour: (%d,%d)", [nextCell x], [nextCell y] );
			[self knockDownWallWithCell:currCell neighbour:nextCell];
			
			[stack addObject:currCell];
			currCell = nextCell;
			visitedCells++;
		}
	}
	endCell = currCell;
	
	
	// add the stack to our bezier path
	NSEnumerator * en = [stack objectEnumerator];
	while ( currCell = [en nextObject] )
	{
		[camSpline addNode:[SBVector3D vector3DWithValuesX:-[currCell x] y:0 z:-[currCell y]]];
	}
	// add the last cell as it won't be on the stack
	[camSpline addNode:[SBVector3D vector3DWithValuesX:-[endCell x] y:0 z:-[endCell y]]];
	// build the spline
	[camSpline buildSpline];
}

- (NSArray *)getIntactNeighboursOfCell:(SBCell *)_cell
{
	int x = [_cell x];
	int y = [_cell y];
	
	SBCell * neighbour;
	NSMutableArray * neighbours = [[NSMutableArray alloc] init];
	
	if ( x > 0 )
	{
		neighbour = [self getCellAtX:x-1 y:y];
		if ( [neighbour wallsIntact] )
		{
			[neighbours addObject:neighbour];
		}
	}
	
	if ( y > 0 )
	{
		neighbour = [self getCellAtX:x y:y-1];
		if ( [neighbour wallsIntact] )
		{
			[neighbours addObject:neighbour];
		}
	}
	
	if ( x < size-1 )
	{
		neighbour = [self getCellAtX:x+1 y:y];
		if ( [neighbour wallsIntact] )
		{
			[neighbours addObject:neighbour];
		}
	}
	
	if ( y < size-1 )
	{
		neighbour = [self getCellAtX:x y:y+1];
		if ( [neighbour wallsIntact] )
		{
			[neighbours addObject:neighbour];
		}
	}
	
	return neighbours;
}

- (void)knockDownWallWithCell:(SBCell *)_currCell neighbour:(SBCell *)_nextCell
{
	if ( [_currCell x] == [_nextCell x] )
	{ // we are working a column
		if ( [_currCell y] + 1 == [_nextCell y] )
		{ // next cell is below us
			[_currCell setDown:NO];
			[_nextCell setUp:NO];
		}
		else if ( [_currCell y] - 1 == [_nextCell y] )
		{
			[_currCell setUp:NO];
			[_nextCell setDown:NO];
		}
		else
		{
			//NSassert( 0 && @"Wall knockdown error" );
		}
	}
	else if ( [_currCell y] == [_nextCell y] )
	{ // we are working a row
		if ( [_currCell x] + 1 == [_nextCell x] )
		{ // next cell is right of us
			[_currCell setRight:NO];
			[_nextCell setLeft:NO];
		}
		else if ( [_currCell x] - 1 == [_nextCell x] )
		{ // next cell is left of us
			[_currCell setLeft:NO];
			[_nextCell setRight:NO];
		}
		else
		{
			//NSassert( 0 && @"Wall knockdown error" );
		}
	}
}

- (SBCell *) getCellAtX:(int)_x y:(int)_y
{
	// ensure we aren't indexing outside our grid
	assert( (_x * size + _y) < size * size && @"Attempt to index cell outside grid" );
	
	return [cells objectAtIndex:( _x * size + _y )];
}

- (NSArray *)cells
{
	return cells;
}

- (void) debugMaze
{
	int x, y;
	
	// first just debug each cell
	/*for ( x = 0; x < size; x++ )
	{
		for ( y = 0; y < size; y++ )
		{
			SBCell * currCell = [self getCellAtX:x y:y];
			NSLog(@"Cell(%d,%d): left: %d right: %d up: %d down: %d",
				  [currCell x], [currCell y],
				  [currCell left], [currCell right],
				  [currCell up], [currCell down]);
		}
	}*/
	
	// now represent this as a top down view
	NSMutableString * line1 = [[NSMutableString alloc] init];
	NSMutableString * line2 = [[NSMutableString alloc] init];
	NSMutableString * line3 = [[NSMutableString alloc] init];
	
	for ( y = 0; y < size; ++y )
	{
		for ( x = 0; x < size; ++x )
		{
			SBCell * currCell = [self getCellAtX:x y:y];
			[line1 appendFormat:@"*%s*", [currCell up] ? "-" : " " ];
			[line2 appendFormat:@"%s%s%s",
				[currCell left] ? "|" : " ",
				currCell == startCell ? "S" : currCell == endCell ? "E" : " ",
				[currCell right] ? "|" : " " ];
			[line3 appendFormat:@"*%s*", [currCell down] ? "-" : " " ];
		}
		
		NSLog( line1 ); [line1 setString:@""];
		NSLog( line2 ); [line2 setString:@""];
		NSLog( line3 ); [line3 setString:@""];
	}
	NSLog( @" " );
}

- (void) addMazeToScene:(SBScene *)_scene
{	
	SBColladaLoader * loader = [[SBColladaLoader alloc] init];
	
	SBSceneEntity * wall = [loader loadObjectWithPath:@"wall.dae"];
	SBSceneEntity * wallWithHole = [loader loadObjectWithPath:@"wallWithHole.dae"];
	SBSceneEntity * floor = [loader loadObjectWithPath:@"floor.dae"];
	SBSceneEntity * startFloor = [loader loadObjectWithPath:@"startFloor.dae"];
	SBSceneEntity * endFloor = [loader loadObjectWithPath:@"endFloor.dae"];
	
	SBSceneEntity * pellet = [loader loadObjectWithPath:@"pellet.dae"];
	[pellet setCollidable:NO];
	
	SBSceneEntity * retreatingPowerUp = [loader loadObjectWithPath:@"retreatingPowerUp.dae"];
	[retreatingPowerUp setName:@"retreatingPowerUp"];
	[retreatingPowerUp setCollidable:NO];
	
	SBSceneEntity * bouncingPowerUp = [loader loadObjectWithPath:@"bouncingPowerUp.dae"];
	[bouncingPowerUp setName:@"bouncingPowerUp"];
	[bouncingPowerUp setCollidable:NO];
	
	SBSceneEntity * elevatePowerUp = [loader loadObjectWithPath:@"elevatePowerUp.dae"];
	[elevatePowerUp setName:@"elevatedPowerUp"];
	[elevatePowerUp setCollidable:NO];
	
	SBSceneEntity * eatingPowerUp = [loader loadObjectWithPath:@"eatingPowerUp.dae"];
	[eatingPowerUp setName:@"eatingPowerUp"];
	[eatingPowerUp setCollidable:NO];
	
	// use these vectors to position each wall in local cell space
	SBVector3D * up    = [[SBVector3D alloc] initWithValuesX: 0 y: 0.5 z: 0.5];
	SBVector3D * down  = [[SBVector3D alloc] initWithValuesX: 0 y: 0.5 z:-0.5];
	SBVector3D * left  = [[SBVector3D alloc] initWithValuesX: 0.5 y: 0.5 z: 0];
	SBVector3D * right = [[SBVector3D alloc] initWithValuesX:-0.5 y: 0.5 z: 0];
	
	SBMat4 * rotMat = [[SBMat4 alloc] initWithRotationY:3.145/2]; // rotate 90 deg
	SBMat4 * floorMat = [[SBMat4 alloc] initWithTranslation:[SBVector3D vector3DWithValuesX:0 y:0 z:0]];
		
	NSEnumerator * en = [cells objectEnumerator];
	SBCell * currCell;
	while ( currCell = [en nextObject] )
	{
		SBSceneEntity * cell = [[SBSceneEntity alloc] init];
		
		SBSceneEntity * newFloor;
		if ( currCell == startCell )
		{
			newFloor = [startFloor copy];
		}
		else if ( currCell == endCell )
		{
			newFloor = [endFloor copy];
		}
		else
		{
			newFloor = [floor copy];
		}
		[newFloor setLocalTransform:floorMat];
		[cell addChild:newFloor];

		SBSceneEntity * newWall;
		BOOL wallType = ( rand() % 10 ) < 7; // randomly place a wall with windows
		
		if ( [currCell up] && ![currCell upDone] )
		{
			newWall = wallType ? [wall copy] : [wallWithHole copy];
			[newWall setLocalTransform:rotMat];
			[[newWall localTransform] setTranslation:up];
			[cell addChild:newWall];
			[self setWallDoneWithCell:currCell wall:1];
		}
		
		if ( [currCell down] && ![currCell downDone] )
		{
			newWall = wallType ? [wall copy] : [wallWithHole copy];
			[newWall setLocalTransform:rotMat];
			[[newWall localTransform] setTranslation:down];
			[cell addChild:newWall];
			[self setWallDoneWithCell:currCell wall:2];
		}
		
		if ( [currCell left] && ![currCell leftDone] )
		{
			newWall = wallType ? [wall copy] : [wallWithHole copy];
			[[newWall localTransform] setTranslation:left];
			[cell addChild:newWall];
			[self setWallDoneWithCell:currCell wall:3];
		}
		
		if ( [currCell right] && ![currCell rightDone] )
		{
			newWall = wallType ? [wall copy] : [wallWithHole copy];
			[[newWall localTransform] setTranslation:right];
			[cell addChild:newWall];
			[self setWallDoneWithCell:currCell wall:4];
		}
		
		// currCell stores x and y cell indices although these are really coords on the x, z plane
		SBVector3D * translation = [SBVector3D vector3DWithValuesX:-[currCell x] y:0 z:-[currCell y]];
		[cell setLocalTransform:[SBMat4 mat4WithTranslation:translation]];
		
		// add a pellet or a powerup
		SBSceneEntity * newPellet;
		BOOL isPowerUp = rand() % 10 > 8;
		
		if ( isPowerUp )
		{
			float pelletType = rand() % 10;
			if ( pelletType >= 2 && pelletType < 4 )
			{
				newPellet = [retreatingPowerUp copy];
			}
			else if ( pelletType >= 4 && pelletType < 6 )
			{
				newPellet = [bouncingPowerUp copy];
			}
			else if ( pelletType >= 6 && pelletType < 8 )
			{
				newPellet = [elevatePowerUp copy];
			}
			else
			{
				newPellet = [eatingPowerUp copy];
			}
		}
		else
		{
			newPellet = [pellet copy];
		}
		
		translation = [SBVector3D vector3DWithValuesX:0 y:0.25 z:0];
		[newPellet setLocalTransform:[SBMat4 mat4WithTranslation:translation]];
		[cell addChild:newPellet];
		
		if ( isPowerUp )
		{
			[_scene addPowerUp:newPellet];
		}
		else
		{
			[_scene addPellet:newPellet];
		}
		
		[cell setName:[NSString stringWithFormat:@"Cell(%d,%d)", [currCell x], [currCell y]]];
		
		[_scene addSceneEntity:cell];
	}
}

// function to avoid drawing the same wall twice
// wall 1 = up, 2 = down, 3 = left, 4 = right
- (void)setWallDoneWithCell:(SBCell *)currCell wall:(int)dir
{
	SBCell * neighbour;
	
	int x = [currCell x], y = [currCell y];
	
	// set the curr cell wall done
	switch (dir)
	{
		case 1:
			[currCell setUpDone:YES];
			if ( y > 0 )
			{
				neighbour = [self getCellAtX:x y:y-1];
				if (neighbour)
					[neighbour setDownDone:YES];
			}
			break;
			
		case 2:
			if ( y < size-1 )
			{
				[currCell setDownDone:YES];
				neighbour = [self getCellAtX:x y:y+1];
				if (neighbour)
					[neighbour setUpDone:YES];
			}
			break;
		case 3:
			if ( x > 0 )
			{
				[currCell setLeftDone:YES];
				neighbour = [self getCellAtX:x-1 y:y];
				if (neighbour)
					[neighbour setRightDone:YES];
			}
			break;
		case 4:
			if ( x < size-1 )
			{
				[currCell setRightDone:YES];
				neighbour = [self getCellAtX:x+1 y:y];
				if (neighbour)
					[neighbour setLeftDone:YES];
			}
			break;
	}
}

- (void)setCameraToStartPosition:(SBCamera *)camera
{
	// set the camera position to the start cell
	SBVector3D * viewPos = [SBVector3D vector3DWithValuesX:-[startCell x] y:0.5 z:-[startCell y]];
	
	[camera setViewPos:viewPos collidingWithOctree:NO];
	[camera setViewDir:[SBVector3D vector3DWithValuesX:[startCell x] - [secondCell x]
													 y:0
													 z:[startCell y] - [secondCell y]]];
}

@end

