//
//  SBNonuniformSpline.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBNonuniformSpline.h"
#import "SBMat4.h"
#import "SBVector3D.h"
#import "SBVector4D.h"

@implementation SBSplineData

- (id) init
{
	if ( self = [super init] )
	{
		position = [[SBVector3D alloc] init];
		velocity = [[SBVector3D alloc] init];
		distance = 0.f;
		return self;
	}
	return nil;
}

- (id) initWithVector3D:(SBVector3D *)pos
{
	if ( self = [super init] )
	{
		position = [[SBVector3D alloc] initWithVector3D:pos];
		velocity = [[SBVector3D alloc] init];
		distance = 0.f;
		return self;
	}
	return nil;
}

- (void) dealloc
{
	[position release];
	[velocity release];
	[super dealloc];
}

- (void)setPosition:(SBVector3D *)newPosition
{
	[position assign:newPosition];
}

- (void)setVelocity:(SBVector3D *)newVelocity
{
	[velocity assign:newVelocity];
}

- (void)setDistance:(float)newDistance
{
	distance = newDistance;
}

- (SBVector3D *)position
{
	return position;
}

- (SBVector3D *)velocity
{
	return velocity;
}

- (float) distance
{
	return distance;
}

@end

@implementation SBNonuniformSpline

- (id) init
{
	if ( self = [super init] )
	{
		nodes = [[NSMutableArray alloc] init];
		return self;
	}
	return nil;
}

- (id) initWithNodes:(NSArray *)newNodes
{
	if ( self = [super init] )
	{
		nodes = [[NSMutableArray alloc] init];

		NSEnumerator * en = [newNodes objectEnumerator];
		NSArray * obj;
		while ( obj = [en nextObject] )
		{
			[self addNode:[SBVector3D vector3DWithValuesX:[[obj objectAtIndex:0] floatValue]
														y:[[obj objectAtIndex:1] floatValue]
														z:[[obj objectAtIndex:2] floatValue]]];
		}
		
		[self buildSpline];
		return self;
	}
	return nil;
}

- (void) dealloc
{
	[nodes release];
	[super dealloc];
}

- (int)numNodes
{
	return [nodes count];
}

- (SBVector3D *)getStartVelocity:(int)index
{
	SBVector3D * temp = [[SBVector3D alloc] initWithVector3D:
		[[[[[nodes objectAtIndex:index+1] position] subtractVector:[[nodes objectAtIndex:index]
			position]] multiplyByFloat:3.f] divideByFloat:[[nodes objectAtIndex:index] distance]]];
	
	return [[temp subtractVector:[[nodes objectAtIndex:index+1] velocity]] multiplyByFloat:0.5f];
}

- (SBVector3D *)getEndVelocity:(int)index
{
	SBVector3D * temp = [[SBVector3D alloc] initWithVector3D:
		[[[[[nodes objectAtIndex:index] position] subtractVector:[[nodes objectAtIndex:index-1]
			position]] multiplyByFloat:3.f] divideByFloat:[[nodes objectAtIndex:index-1] distance]]];
	
	return [[temp subtractVector:[[nodes objectAtIndex:index-1] velocity]] multiplyByFloat:0.5f];
}

- (SBVector3D *)getPositionOnCubicWithStartPos:(SBVector3D *)startPos
										startVel:(SBVector3D *)startVel
										endPos:(SBVector3D *)endPos
										endVel:(SBVector3D *)endVel
										atTime:(float)time
{
	SBMat4 * m = [[SBMat4 alloc] initWithVec3Col0:startPos col1:endPos col2:startVel col3:endVel];
	
	//NSLog( @"G:" );
	//[m logValuesAsRows];
	
	SBMat4 * hermiteTransposed = [[SBMat4 alloc]
		initWithVec4Col0:[SBVector4D vector4DWithValuesX: 2.f y:-2.f z: 1.f w: 1.f] 
					col1:[SBVector4D vector4DWithValuesX:-3.f y: 3.f z:-2.f w:-1.f]
					col2:[SBVector4D vector4DWithValuesX: 0.f y: 0.f z: 1.f w: 0.f]
					col3:[SBVector4D vector4DWithValuesX: 1.f y: 0.f z: 0.f w: 0.f]];
	
	//NSLog( @"H:" );
	//[hermiteTransposed logValuesAsRows];
	
	[m assign:[m multiplyByMat4:hermiteTransposed]];
	
	//NSLog( @"Result:" );
	//[m logValuesAsRows];	

	SBVector4D * timeVector = [[SBVector4D alloc]
		initWithValuesX:time*time*time y:time*time z:time w:1.f];

	SBVector4D * result = [m multiplyByVector4D:timeVector];
	
	return [SBVector3D vector3DWithValuesX:[result x] y:[result y] z:[result z]];
}

- (void) addNode:(SBVector3D *)pos
{
	if ( [nodes count] == 0 )
	{
		maxDistance = 0.f;
	}
	else
	{
		[[nodes lastObject] setDistance: [[[[nodes lastObject] position] subtractVector:pos] length]];
		maxDistance += [[nodes lastObject] distance];
	}
	
	[nodes addObject:[[SBSplineData alloc] initWithVector3D:pos]];
}

- (void) buildSpline
{
	int i;
	for (i = 1; i < [nodes count] - 1; ++i)
	{
		SBVector3D * nextNodeDirection = [[[nodes objectAtIndex:i+1] position] subtractVector:[[nodes objectAtIndex:i] position]];
		[nextNodeDirection normalise];
		SBVector3D * prevNodeDirection = [[[nodes objectAtIndex:i-1] position] subtractVector:[[nodes objectAtIndex:i] position]];
		[prevNodeDirection normalise];
		
		[[nodes objectAtIndex:i] setVelocity: [nextNodeDirection subtractVector:prevNodeDirection]];
		[[[nodes objectAtIndex:i] velocity] normalise];
	}
	
	[[nodes objectAtIndex:0] setVelocity:[self getStartVelocity:0]];
	[[nodes lastObject] setVelocity:[self getEndVelocity:[nodes count]-1]];
	
	/*
	NSEnumerator * en = [nodes objectEnumerator];
	id node;
	while ( node = [en nextObject] )
	{
		SBVector3D * pos = [node position];
		SBVector3D * vel = [node velocity];
		NSLog( @"pos: [ %f, %f, %f ] vel: [ %f, %f, %f ]", [pos x], [pos y], [pos z], [vel x], [vel y], [pos z] );
	}*/
}

- (SBVector3D *)getPositionAtTime:(float)time
{
	float distance = time * maxDistance;
	float currentDistance = 0.f;
	int i = 0;
	while ( currentDistance + [[nodes objectAtIndex:i] distance] < distance )
	{
		currentDistance += [[nodes objectAtIndex:i] distance];
		++i;
	}
	
	// NSLog( @"time: %f, idx: %d", time, i );
	
	SBSplineData * segmentStartNode = [nodes objectAtIndex:i];
	SBSplineData * segmentEndNode = [nodes objectAtIndex:i+1];
	
	float t = distance - currentDistance;
	t /= [[nodes objectAtIndex:i] distance]; // scale t in range 0 - 1
	
	// NSLog( @"time: %f, t: %f", time, t );
	
	SBVector3D * startVel = [[segmentStartNode velocity] multiplyByFloat:[segmentStartNode distance]];
	
	//NSLog( @"segmentStartNode: " );
	//[[segmentStartNode position] logValues];
	
	SBVector3D * endVel = [[segmentEndNode velocity] multiplyByFloat:[segmentStartNode distance]];
	
	//NSLog( @"segmentEndNode: " );
	//[[segmentEndNode position] logValues];
	
	return [self getPositionOnCubicWithStartPos:[segmentStartNode position] startVel:startVel endPos:[segmentEndNode position] endVel:endVel atTime:t];
}

@end