//
//  SBOctreeNode.m
//  SBGame
//
//  Created by Stuart Bryson on 25/09/05.
//  Copyright 2005 __MyCompanyName__. All rights reserved.
//

#import "SBOctreeNode.h"

#import "AABB.h"
#import "SBVector3D.h"

#import "SBScene.h"
#import "SBSceneEntity.h"

@implementation SBOctreeNode

static maxDepth = 0;

+ (int)maxDepth
{
	return maxDepth;
}

+ (void)setMaxDepth:(int)_maxDepth
{
	maxDepth = _maxDepth;
}

- (id) init
{
	if ( self = [super init] )
	{
		parent = nil;
		children = [[NSMutableArray alloc] init];
		extents = [[AABB alloc] init];
		objects = [[NSMutableArray alloc] init];
		return self;
	}
	return nil;
}

- (void) allocQuadChildren
{
	// calculate the extents for each of the 8 children
	SBVector3D * pMin = [extents min]; // parent min
	SBVector3D * pMax = [extents max];
	SBVector3D * pCen = [[[pMax addVector:pMin] divideByFloat:2.f] retain];
	
	AABB * child;
	
	// child a; min = min, min, min; max = half, max, half
	child = [AABB AABBWithMin:pMin
						  max:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMax y] z:[pCen z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child b; min = half, min, min; max = max, max, half
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMin y] z:[pMin z]]
						  max:[SBVector3D vector3DWithValuesX:[pMax x] y:[pMax y] z:[pCen z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child c; min = min, min, half; max = half, max, max
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pMin x] y:[pMin y] z:[pCen z]]
						  max:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMax y] z:[pMax z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child d; min = half, min, half; max = max, max, max
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMin y] z:[pCen z]]
						  max:pMax];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	[pCen release];
}

- (void) allocOctChildren
{
	// calculate the extents for each of the 8 children
	SBVector3D * pMin = [extents min];
	SBVector3D * pMax = [extents max];
	SBVector3D * pCen = [[[pMax addVector:pMin] divideByFloat:2.f] retain];
	
	AABB * child;
	
	// child a; min = min, half, half; max = half, max, max
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pMin x] y:[pCen y] z:[pCen z]]
						  max:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMax y] z:[pMax z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child b; min = half, half, half; max = max, max, max
	child = [AABB AABBWithMin:pCen
						  max:pMax];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child c; min = min, min, half; max = half, half, max
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pMin x] y:[pMin y] z:[pCen z]]
						  max:[SBVector3D vector3DWithValuesX:[pCen x] y:[pCen y] z:[pMax z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child d; min = half, min, half; max = max, half, max
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMin y] z:[pCen z]]
						  max:[SBVector3D vector3DWithValuesX:[pMax x] y:[pCen y] z:[pMax z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child e; min = min, half, min; max = half, max, half
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pMin x] y:[pCen y] z:[pMin z]]
						  max:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMax y] z:[pCen z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child f; min = half, half, min; max = max, max, half
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pCen x] y:[pCen y] z:[pMin z]]
						  max:[SBVector3D vector3DWithValuesX:[pMax x] y:[pMax y] z:[pCen z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child g; min = min, min, min; max = half, half, half
	child = [AABB AABBWithMin:pMin
						  max:pCen];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	// child h; min = half, min, min; max = max, half, half
	child = [AABB AABBWithMin:[SBVector3D vector3DWithValuesX:[pCen x] y:[pMin y] z:[pMin z]]
						  max:[SBVector3D vector3DWithValuesX:[pMax x] y:[pCen y] z:[pCen z]]];
	[children addObject:[[SBOctreeNode alloc] initWithParent:self extents:child depth:depth+1]];
	
	[pCen release];
}

- (void)allocChildren
{
	// adapatively determine if this node should be an oct node or a quad node
	//  - this method could also use a binary node
	//  - for now we will just test if the height is less than half the depth or width
	
	SBVector3D * min = [extents min];
	SBVector3D * max = [extents max];
	
	float bbWidth  = [max x] - [min x];
	float bbHeight = [max y] - [min y];
	float bbDepth  = [max z] - [min z];
	
	if ( bbHeight < bbWidth / 2 || bbHeight < bbDepth / 2 )
	{
		//NSLog( @"Child is quad" );
		[self allocQuadChildren];
	}
	else
	{
		//NSLog( @"Child is oct" );
		[self allocOctChildren];
	}
}

- (id)initWithParent:(SBOctreeNode *)_parent extents:(AABB *)_extents depth:(int)_depth;
{
	if ( self = [super init] )
	{
		depth = _depth;
		
		parent = _parent;
		extents = [[_extents copy] retain];
		
		children = [[NSMutableArray alloc] init];
		if ( depth < maxDepth )
		{
			[self allocChildren];
		}
		
		objects = [[NSMutableArray alloc] init];
		
		return self;
	}
	
	return nil;
}

- (BOOL)addObject:(SBSceneEntity *)entity bestFit:(BOOL)bestFit
{
	BOOL added = NO;
	
	if ( bestFit )
	{
		// must try to add it to the child nodes first
		NSEnumerator * en = [children objectEnumerator];
		SBOctreeNode * child;
		while ( child = [en nextObject] )
		{
			added = [child addObject:entity bestFit:YES];
			if ( added )
			{
				return YES;
			}
		}
	}

	if ( [extents containsAABB:[entity worldAABB]] )
	{
		[entity setOctNode:self];
		[objects addObject:entity];
		return YES;
	}
	return NO;
}

// this remove object is not recursive! The entity knows which octNode it is in
- (void)removeObject:(SBSceneEntity *)entity
{
	[objects removeObject:entity];
}

- (void) dealloc
{
	if ( parent )
	{
		[parent release];
	}
	[children release];
	[extents release];
	[objects release];
	[super dealloc];
}

- (SBSceneEntity *)collideWith:(AABB *)aabb
{
	// if the aabb does not intersect this node return NO
	if ( ![extents intersects:aabb] )
	{
		return NO;
	}
	
	// if we do intersect, test against the objects in this node
	NSEnumerator * objEn = [objects objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [objEn nextObject] )
	{
		if ( [[entity worldAABB] intersects:aabb] )
		{
			return entity;
		}
	}
	
	
	// if there is no collision, test against all children of this node
	NSEnumerator * octEn = [children objectEnumerator];
	SBOctreeNode * child;
	SBSceneEntity * collidedWith;
	while ( child = [octEn nextObject] )
	{
		if ( collidedWith = [child collideWith:aabb] )
		{
			return collidedWith;
		}
	}
	
	// we have tested the object and the children and there is no collision
	return nil;
}

- (int) depth { return depth; }
- (SBOctreeNode *) parent { return parent; }
- (NSMutableArray *) children { return children; }
- (AABB *) extents { return extents; }
- (NSMutableArray *) objects { return objects; }

- (void) logValues
{
	NSLog( @"Depth: %d Object Count: %d", depth, [objects count] );
	
	NSEnumerator * en = [children objectEnumerator];
	SBOctreeNode * child;
	while ( child = [en nextObject] )
	{
		[child logValues];
	}
}
@end
