//
//  SBPathPoints.m
//  SBGame
//
//  Created by Stuart Bryson on 24/10/05.
//  Copyright 2005 __MyCompanyName__. All rights reserved.
//

#import "SBPathPoints.h"
#import "SBPathPoint.h"
#import "SBCell.h"
#import "SBVector3D.h"

@implementation SBPathPoints

- (id) initWithCells:(NSArray *)cells
{
	
	if ( self = [super init] )
	{
		// firstly create all our points
		points = [[NSMutableArray alloc] init];
		unsigned int i;
		for ( i = 0; i < [cells count]; ++i )
		{
			[points addObject:[[[SBPathPoint alloc] init] autorelease]];
		}
		
		unsigned int x, y, size = sqrtl([cells count]);
		
		for ( x = 0; x < size; ++x )
		{
			for ( y = 0; y < size; ++y )
			{
				SBCell * currCell = [cells objectAtIndex:( x * size + y )];
				SBPathPoint * currPoint = [points objectAtIndex: ( x * size + y )];
				
				// set the translation to be the position of the cell ( this correspondes with Maze generator )
				[currPoint setTranslation:[SBVector3D vector3DWithValuesX:-[currCell x] y:0 z:-[currCell y]]];
				
				if ( x > 0 && ![currCell left] )
				{
					[currPoint addNeighbour:[points objectAtIndex: ( (x-1) * size + y )]];
				}
				
				if ( y > 0 && ![currCell up] )
				{
					[currPoint addNeighbour:[points objectAtIndex: ( x * size + y-1 )]];
				}
				
				if ( x < size-1 && ![currCell right] )
				{
					[currPoint addNeighbour:[points objectAtIndex: ( (x+1) * size + y )]];
				}
				
				if ( y < size-1 && ![currCell down] )
				{
					[currPoint addNeighbour:[points objectAtIndex: ( x * size + y+1 )]];
				}
			}
		}
		return self;
	}
	return nil;
}

- (void)dealloc
{
	[points release];
	[super dealloc];
}

- (NSArray *)points { return points; }

- (SBPathPoint *)getBestCostNodeFrom:(NSArray *)nodes
{
	NSEnumerator * en = [nodes objectEnumerator];
	SBPathPoint * currPoint, * bestPoint;
	
	bestPoint = [en nextObject];
	
	while ( currPoint = [en nextObject] )
	{
		if ( [currPoint f] < [bestPoint f] )
		{
			bestPoint = currPoint;
		}
	}
	return bestPoint;
}

- (BOOL)betterNode:(SBPathPoint *)node inList:(NSArray *)list
{
	NSEnumerator * en = [list objectEnumerator];
	SBPathPoint * currPoint;
	
	while ( currPoint = [en nextObject] )
	{
		if ( [[currPoint translation] equals:[node translation]] && [currPoint f] <= [node f] )
		{
			return YES;
		}
	}
	return NO;
}

- (NSArray *)generatePathFromDestinationNode:(SBPathPoint *)dest
{
	NSMutableArray * path = [[[NSMutableArray alloc] init] autorelease];
	
	[path addObject:dest];
	SBPathPoint * currPoint = dest;
	while ( currPoint = [currPoint parent] )
	{
		[path insertObject:currPoint atIndex:0];
	}
	return path;	
}

- (SBPathPoint *)findFarthestFrom:(SBVector3D *)offPath
{
	NSEnumerator * en = [points objectEnumerator];
	
	SBPathPoint * farthestPoint = [en nextObject];
	float farthestDist = [offPath distanceTo:[farthestPoint translation]];
	
	SBPathPoint * currPoint;
	while ( currPoint = [en nextObject] )
	{
		float currDist = [offPath distanceTo:[currPoint translation]];
		if ( currDist > farthestDist )
		{
			farthestDist = currDist;
			farthestPoint = currPoint;
		}
	}
	return farthestPoint;
}

- (SBPathPoint *)findClosestTo:(SBVector3D *)offPath
{
	NSEnumerator * en = [points objectEnumerator];
	
	SBPathPoint * bestPoint = [en nextObject];
	float bestDist = [offPath distanceTo:[bestPoint translation]];
	
	SBPathPoint * currPoint;
	while ( currPoint = [en nextObject] )
	{
		float currDist = [offPath distanceTo:[currPoint translation]];
		if ( currDist < bestDist )
		{
			bestDist = currDist;
			bestPoint = currPoint;
		}
	}
	return bestPoint;
}

- (NSArray *)findPathFromVector:(SBVector3D *)from toVector:(SBVector3D *)to
{
	// find the closest path point to each of the from and to vectors
	SBPathPoint * fromPathPoint = [self findClosestTo:from];
	SBPathPoint * toPathPoint = [self findClosestTo:to];
	
	return [self findPathFromPathPoint:fromPathPoint toPathPoint:toPathPoint];
}

- (NSArray *)findPathFromPathPoint:(SBPathPoint *)fromPoint toPathPoint:(SBPathPoint *)toPoint
{
	NSMutableArray * open = [[NSMutableArray alloc] init];
	NSMutableArray * closed = [[NSMutableArray alloc] init];
	
	// add the from point to the open list
	[open addObject:fromPoint];
	
	while ( [open count] > 0 )
	{
		SBPathPoint * currPoint = [self getBestCostNodeFrom:open];		
		[open removeObject:currPoint];
		
		NSEnumerator * en = [[currPoint neighbours] objectEnumerator];
		SBPathPoint * successor;
		while ( successor = [en nextObject] )
		{
			SBPathPoint * newPathPoint = [successor copy];
			[newPathPoint setParent:currPoint];
			
			if ( [[newPathPoint translation] equals:[toPoint translation]] )
			{
				return [self generatePathFromDestinationNode:newPathPoint];
			}
			
			[newPathPoint setG:[currPoint g] + [[currPoint translation] distanceTo:[newPathPoint translation]]];
			[newPathPoint setH:[[currPoint translation] distanceTo:[toPoint translation]]];
			
			if ( [self betterNode:newPathPoint inList:open] )
				continue;
			if ( [self betterNode:newPathPoint inList:closed] )
				continue;
			
			[open addObject:newPathPoint];
		}
		[closed addObject:currPoint];
	}
	return nil;
}

- (void) debugPoints
{	
	NSEnumerator * en = [points objectEnumerator];
	SBPathPoint * point;
	while ( point = [en nextObject] )
	{
		unsigned int nCount = [[point neighbours] count];
		NSLog( @"Point has %d neighbours", nCount );
	}
}

@end