//
//  SBScene.h
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import <Cocoa/Cocoa.h>

#import <OpenGL/OpenGL.h>
#import <OpenGL/gl.h>
#import <OpenGL/glu.h>

// cant forward declare enemy due to declared enum used in this header file
#import "SBEnemy.h"

@class SBCamera, SBSceneEntity, AABB, SBOctreeNode, SBPathPoints, SBVector3D, SBLight;

// should this be the delegate for keydown events of the SBGLView?
// that way we can filter key events based on game logic more easily

// startGame - call endGame first?
// reset player to start loc
// remove all enemies
// reset remaining lives
// start the spawn timer

// endGame
// disable camera movement
// disable spawn timer

@interface SBScene : NSObject {
	
	NSMutableArray * sceneEntities;
	NSMutableArray * enemies;
	NSMutableArray * pellets;
	NSMutableArray * powerUps;
	NSMutableArray * lights;
	
	SBVector3D * playerStartLoc;
	SBVector3D * enemySpawnLoc;
	
	SBCamera * camera;
	AABB * aabb;
	SBOctreeNode * rootOctreeNode;

	SBPathPoints * pathPoints;

	CFAbsoluteTime startTime;

	unsigned int size; // grid size of the maze
	int remainingLives;
	unsigned int maxEnemies;
	int score;
	
	NSTimer * updateTimer;
	NSTimer * spawnTimer;
	NSTimeInterval spawnTimeInterval;
	
	// loaded assets
	SBEnemy * masterEnemy;
	NSSound * playerDieSnd, * eatGhostSnd, * ghostStartsSnd,
		* eatPelletBSnd, * eatPelletB2Snd, * powerUpSnd,
		* levelClearSnd, * gameOverSnd;
	BOOL playPelletB2;
	
	BOOL active;
	
	BOOL strobe, lightPulseUp, strobeRed, strobeGreen, strobeBlue; // lighting effects
}

- (void) startGame;
- (void) endGame;

- (void) clear;

- (void) startUpdateTimer;
- (void) updateScene:(NSTimer *)timer;

- (void) addSceneEntity: (SBSceneEntity *)_sceneEntity;
- (NSMutableArray *) sceneEntities;
- (unsigned int) numEntities;
- (void) updateSceneEntities;
- (void) updateSceneEntityWorldTransforms;
- (SBSceneEntity *) getSceneEntityWithName:(NSString *)entityName;
- (void) toggleShaderOnSceneEntityWithName:(NSString *)entityName;

- (void) setEnemySpawnLoc:(SBVector3D *)_spawnLoc;
- (void) setMasterEnemy:(SBEnemy *)_enemy;
- (void) addEnemy: (SBEnemy *)_enemy;
- (void) removeEnemy:(SBEnemy *)_enemy;
- (void) removeAllEnemies;
- (void) spawnEnemy:(NSTimer *)timer;
- (void) updateEnemies;
- (void) setEnemyStates:(EnemyState)_state;
- (void) setSpawnTimeInterval:(float)_interval;
- (void) startEnemySpawnTimer;
- (void) resetEnemySpawnTimer;

- (void) addPellet:(SBSceneEntity *)_pellet;
- (void) removePellet:(SBSceneEntity *)_pellet;
- (void) removeAllPellets;

- (void) addPowerUp:(SBSceneEntity *)_powerUp;
- (void) removePowerUp:(SBSceneEntity *)_powerUp;
- (void) removeAllPowerUps;

- (void) collideWithObjects;
- (void) collideWithEnemies;
- (void) collideWithPellets;
- (void) collideWithPowerUps;

- (void)setPathPoints:(SBPathPoints *)_pathPoints;
- (SBPathPoints *)pathPoints;

- (void) setPlayerStartLoc:(SBVector3D *)_startLoc;
- (void) setCamera:(SBCamera *)_camera;
- (SBCamera *) camera;

- (void) setAABB:(AABB *)_aabb;
- (SBOctreeNode *) rootOctreeNode;
- (void) calculateOctree;

- (void) setSize:(unsigned int)_size;
- (unsigned int) size;
- (int) score;
- (int) remainingLives;

- (void) setupLights;
- (void) resetLights;
- (void) updateLights;
- (NSArray *) lights;

- (void) loadSounds;
- (void) loadTextures;

- (void) logOctree;
- (void) logSceneEntities;

@end
