//
//  SBScene.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBScene.h"
#import "SBCamera.h"
#import "SBSceneEntity.h"
#import "SBVector3D.h"
#import "SBVectorRGB.h"
#import "SBLight.h"
#import "AABB.h"
#import "SBMat4.h"
#import "SBOctreeNode.h"
#import "SBDrawable.h"
#import "SBPathPoints.h"
#import "Controller.h"

static const PELLET_POINT = 5;
static const ENEMY_POINT = 20;

@implementation SBScene

#pragma mark ---- Initialisation ----

- (id) init
{
	if ( self = [super init] )
	{
		maxEnemies = 4;
		
		sceneEntities = [[NSMutableArray alloc] init];
		enemies = [[NSMutableArray alloc] initWithCapacity:maxEnemies];
		pellets = [[NSMutableArray alloc] init];
		powerUps = [[NSMutableArray alloc] init];
		
		lights = [[NSMutableArray alloc] init];
		[self setupLights];
		
		playerStartLoc = [[SBVector3D alloc] init];
		enemySpawnLoc = [[SBVector3D alloc] initWithValuesX:0 y:0 z:0];

		aabb = [[AABB alloc] init];

		rootOctreeNode = nil;
		camera = nil;
		masterEnemy = nil;
		spawnTimeInterval = 5;

		[self loadTextures];
		[self loadSounds];
		playPelletB2 = false;

		return self;
	}

	return nil;
}

- (void) dealloc
{	
	[sceneEntities release];
	[enemies release];
	[pellets release];
	[powerUps release];
	[lights release];
	
	[playerStartLoc release];
	[enemySpawnLoc release];
	[aabb release];
	
	if ( rootOctreeNode )
	{
		[rootOctreeNode release];
		rootOctreeNode = nil;
	}
	
	if ( pathPoints )
	{
		[pathPoints release];
		pathPoints = nil;
	}
	
	if ( spawnTimer )
	{
		[spawnTimer release];
		spawnTimer = nil;
	}
	
	if ( updateTimer )
	{
		[updateTimer release];
		updateTimer = nil;
	}
	
	if ( masterEnemy )
	{
		[masterEnemy release];
		masterEnemy = nil;
	}
	
	if ( playerDieSnd )
	{
		[playerDieSnd release];
		playerDieSnd = nil;
	}
	
	if ( eatGhostSnd )
	{
		[eatGhostSnd release];
		eatGhostSnd = nil;
	}
	
	if ( ghostStartsSnd )
	{
		[ghostStartsSnd release];
		ghostStartsSnd = nil;
	}
	
	if ( eatPelletBSnd )
	{
		[eatPelletBSnd release];
		eatPelletBSnd = nil;
	}
	
	if ( eatPelletB2Snd )
	{
		[eatPelletB2Snd release];
		eatPelletB2Snd = nil;
	}
	
	if ( powerUpSnd )
	{
		[powerUpSnd release];
		powerUpSnd = nil;
	}
	
	if ( levelClearSnd )
	{
		[levelClearSnd release];
		levelClearSnd = nil;
	}
	
	if ( gameOverSnd )
	{
		[gameOverSnd release];
		gameOverSnd = nil;
	}
	
	[super dealloc];
}

- (void) clear
{
	[sceneEntities removeAllObjects];
	[enemies removeAllObjects];
	[pellets removeAllObjects];
	[powerUps removeAllObjects];
	[lights removeAllObjects];
	[self setupLights];
	
	if ( rootOctreeNode )
	{
		[rootOctreeNode release];
		rootOctreeNode = nil;
	}
	
	if ( pathPoints )
	{
		[pathPoints release];
		pathPoints = nil;
	}
	
	if ( spawnTimer )
	{
		[spawnTimer invalidate];
		spawnTimer = nil;
	}
	
	if ( updateTimer )
	{
		[updateTimer invalidate];
		updateTimer = nil;
	}
}

#pragma mark ---- Game ----

- (void) startGame
{
	[camera setActive:YES];
	[camera setViewPos:playerStartLoc collidingWithOctree:NO];
	[self removeAllEnemies];
	
	remainingLives = 3;
	score = 0;

	[self startEnemySpawnTimer];
	[self startUpdateTimer];
}

- (void) endGame
{
	[self removeAllEnemies];
	[self removeAllPellets];
	[self removeAllPowerUps];
	
	[camera setActive:NO];
	if ( spawnTimer )
	{
		[spawnTimer invalidate];
		spawnTimer = nil;
	}
	
	if ( updateTimer )
	{	
		[updateTimer invalidate];
		updateTimer = nil;
	}
}

- (void) startUpdateTimer
{
	updateTimer = [[NSTimer scheduledTimerWithTimeInterval:0.05
													target:self
												  selector:@selector( updateScene: ) // call this at each interval
												  userInfo:nil repeats:YES] retain];
	
	// add to the event and modal run loops
	[[NSRunLoop currentRunLoop] addTimer:updateTimer forMode:NSEventTrackingRunLoopMode];
	[[NSRunLoop currentRunLoop] addTimer:updateTimer forMode:NSModalPanelRunLoopMode];
}

- (void)updateScene:(NSTimer *)timer
{
	//[scene updateSceneEntities]; // nothing to update currently
	[self collideWithObjects];
	[self updateEnemies];
	[self updateLights];
}

#pragma mark ---- Scene Entites ----

- (void) addSceneEntity: (SBSceneEntity *)_sceneEntity
{
	assert( _sceneEntity != nil );
	[sceneEntities addObject:_sceneEntity];
}
- (NSMutableArray *)sceneEntities { return sceneEntities; }
- (unsigned int)numEntities { return [sceneEntities count]; }

- (void) updateSceneEntities
{
	NSEnumerator * en = [sceneEntities objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity updateWithTime:0];
	}
}

- (void) updateSceneEntityWorldTransforms
{
	NSEnumerator * en = [sceneEntities objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity updateWorldTransform];
	}
	
	NSEnumerator * en2 = [sceneEntities objectEnumerator];
	while ( entity = [en2 nextObject] )
	{
		[entity calculateAABBFromChildren:aabb];
	}
}

// returns the first occurence of a scene entity with the name: entityName
//  - inefficient method, don't call every frame!
- (SBSceneEntity *) getSceneEntityWithName:(NSString *)entityName
{
	NSEnumerator * en = [sceneEntities objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		if ( [[entity name] isEqualToString:entityName] )
		{
			return entity;
		}
		
		// check this entities children
		SBSceneEntity * child = [entity getSceneEntityWithName:entityName];
		if ( child )
		{
			return child;
		}
	}
	return nil;
}

- (void) toggleShaderOnSceneEntityWithName:(NSString *)entityName
{
	SBSceneEntity * entity = [self getSceneEntityWithName:entityName];
	[entity setUseAlternateTexture: [entity useAlternateTexture] ? NO : YES];
}


#pragma mark ---- Enemies ----

- (void) setEnemySpawnLoc:(SBVector3D *)_startLoc
{
	//[enemySpawnLoc assign:_startLoc];
}

- (void) setMasterEnemy:(SBEnemy *)_enemy
{
	if ( masterEnemy )
	{
		[masterEnemy release];
	}
	masterEnemy = [_enemy retain];
}

- (void) setSpawnTimeInterval:(float)_interval
{
	spawnTimeInterval = _interval;
}

- (void) resetEnemySpawnTimer
{
	[spawnTimer invalidate];
	[self startEnemySpawnTimer];
}

- (void) startEnemySpawnTimer
{
	spawnTimer = [NSTimer scheduledTimerWithTimeInterval:spawnTimeInterval
										 target:self
									   selector:@selector( spawnEnemy: )
									   userInfo:nil repeats:YES];
}

- (void) spawnEnemy:(NSTimer *)timer
{
	if ( [enemies count] < maxEnemies )
	{
		SBEnemy * newGhost = [masterEnemy copy];
		if ( newGhost )
		{
			//int x = rand()%size;
			//int z = rand()%size;
			//SBVector3D * translation = [SBVector3D vector3DWithValuesX:-x y:0.2 z:-z];
			//[newGhost setLocalTransform:[SBMat4 mat4WithTranslation:translation]];

			[newGhost setLocalTransform:[SBMat4 mat4WithTranslation:enemySpawnLoc]];
			[newGhost updateWorldTransform]; // careful, the ghost doesn't have a parent entity
			[newGhost setName:@"ghost"];
			
			[ghostStartsSnd play];
			
			[self addEnemy:newGhost];
		}
	}
}

- (void) addEnemy: (SBEnemy *)_enemy
{
	if ( _enemy )
	{
		[enemies addObject:_enemy];
		[sceneEntities addObject:_enemy];
		
		// all enemies live in the root octTree node
		//  - this saves us recalculating the node an enemy should live in each frame
		//  - faster just to draw them since there aren't many
		//  - for this reason, we can't calculate the octree after the enimies have been added
		
		[rootOctreeNode addObject:(SBSceneEntity *)_enemy bestFit:NO];
	}
}

- (void) removeEnemy:(SBEnemy *)_enemy
{
	// need to return to attack to ensure that the timer doesn't get called when the object is released
	[_enemy returnToAttack:nil];
	
	[[_enemy octNode] removeObject:(SBSceneEntity *)_enemy];
	[sceneEntities removeObject:_enemy];
	[enemies removeObject:_enemy];
}

- (void) removeAllEnemies
{
	NSEnumerator * removeEn = [enemies objectEnumerator];
	SBEnemy * enemy;
	while ( enemy = [removeEn nextObject] )
	{
		[self removeEnemy:enemy];
	}
}

- (void) updateEnemies
{
	BOOL allAttacking = YES;
	
	NSEnumerator * en = [enemies objectEnumerator];
	SBEnemy * enemy;
	while ( enemy = [en nextObject] )
	{
		[enemy updateWithTime:0]; // we don't use the time here yet
		allAttacking = [enemy state] == ES_ATTACKING && allAttacking;
	}
	
	if ( strobe && allAttacking )
	{
		[self resetLights];
	}
}

- (void) setEnemyStates:(EnemyState)_state
{
	if ( _state == ES_RETREATING )
	{
		strobe = YES;
		strobeRed = strobeGreen = YES;
		strobeBlue = NO;
	}
	else if ( _state == ES_ELEVATED )
	{
		strobe = YES;
		strobeRed = strobeBlue = YES;
		strobeGreen = NO;
	}
	else if ( _state == ES_BOUNCING )
	{
		strobe = YES;
		strobeGreen = strobeBlue = YES;
		strobeRed = NO;
	}
	
	NSEnumerator * en = [enemies objectEnumerator];
	SBEnemy * enemy;
	while ( enemy = [en nextObject] )
	{
		[enemy setState:_state];
	}
}

#pragma mark ---- Other Entities - Pellets ----

- (void) addPellet:(SBSceneEntity *)_pellet
{
	// pellets are assumed to be already added to the octree and scene entities array
	[pellets addObject:_pellet];
}

- (void) removePellet:(SBSceneEntity *)_pellet
{
	[[_pellet octNode] removeObject:_pellet];
	
	unsigned int idx = [sceneEntities indexOfObject:_pellet];
	if ( idx == NSNotFound )
	{
		NSEnumerator * en = [sceneEntities objectEnumerator];
		SBSceneEntity * entity;
		while ( entity = [en nextObject] )
		{
			[entity recursiveRemoveEntity:_pellet];
		}
	}
	else
	{
		[sceneEntities removeObjectAtIndex:idx];
	}
	
	[pellets removeObject:_pellet];
	
	if ( playPelletB2 )
	{
		[eatPelletB2Snd play];
	}
	else
	{
		[eatPelletBSnd play];
	}
}

- (void) removeAllPellets
{
	NSEnumerator * en = [pellets objectEnumerator];
	SBSceneEntity * pellet;
	while ( pellet = [en nextObject] )
	{
		[self removePellet:pellet];
	}
}

- (void) addPowerUp:(SBSceneEntity *)_powerUp
{
	// powerUps are assumed to be already added to the octree and scene entities array
	[powerUps addObject:_powerUp];
}

- (void) removePowerUp:(SBSceneEntity *)_powerUp
{
	[[_powerUp octNode] removeObject:_powerUp];
	
	unsigned int idx = [sceneEntities indexOfObject:_powerUp];
	if ( idx == NSNotFound )
	{
		NSEnumerator * en = [sceneEntities objectEnumerator];
		SBSceneEntity * entity;
		while ( entity = [en nextObject] )
		{
			[entity recursiveRemoveEntity:_powerUp];
		}
	}
	else
	{
		[sceneEntities removeObjectAtIndex:idx];
	}
	
	[powerUps removeObject:_powerUp];
	
	[powerUpSnd play];
}

- (void) removeAllPowerUps
{
	NSEnumerator * en = [powerUps objectEnumerator];
	SBSceneEntity * powerUp;
	while ( powerUp = [en nextObject] )
	{
		[self removePowerUp:powerUp];
	}
}

#pragma mark ---- Special Collision Handling ----
// note this is for 'non' collideable entities, ie not walls or floors but pellets and ghosts

- (void) collideWithObjects
{
	[self collideWithEnemies];
	[self collideWithPellets];

	// if we have no pellets left then we have finished the game (level?)
	if ( [pellets count] == 0 )
	{
		[levelClearSnd play];
		[self endGame];
	}

	[self collideWithPowerUps];
}

- (void) collideWithEnemies
{
	NSEnumerator * en = [enemies objectEnumerator];
	SBEnemy * enemy;
	AABB * camAABB = [camera worldAABB];
	while ( enemy = [en nextObject] )
	{
		if ( [camAABB intersects:[enemy worldAABB]] )
		{			
			if ( [enemy state] != ES_RETREATING )
			{				
				remainingLives--;
				
				NSString * message = [NSString stringWithFormat:@"Lives left:%d", remainingLives];
				
				[[NSNotificationCenter defaultCenter]
					postNotificationName:@"SBMessage" object:self
						userInfo:[NSDictionary dictionaryWithObject:message forKey:@"message"]];
				
				if ( remainingLives < 0 )
				{
					[gameOverSnd play];
					[self endGame];
					return;
				}
				
				[playerDieSnd play];
				[self resetEnemySpawnTimer];
				
				[self removeAllEnemies];
				[camera setViewPos:playerStartLoc collidingWithOctree:NO];
			}
			else
			{
				[eatGhostSnd play];
				score += ENEMY_POINT;
				[self removeEnemy:enemy];
				[self resetEnemySpawnTimer];
			}
		}
	}
}

- (void) collideWithPellets
{	
	// build up a list of enemies that are in the eating state
	NSMutableArray * eatingEnemies = [[NSMutableArray alloc] initWithCapacity:maxEnemies];
	NSEnumerator * enemyEn = [enemies objectEnumerator];
	SBEnemy * enemy;
	while ( enemy = [enemyEn nextObject] )
	{
		if ( [enemy state] == ES_EATING )
		{
			[eatingEnemies addObject:enemy];
		}
	}
	
	// enumerate all the pellets in the scene, test player and possibly enemy collision against the pellets
	NSEnumerator * pelletEn = [pellets objectEnumerator];
	SBSceneEntity * pellet;
	AABB * camAABB = [camera worldAABB];
	
pelletEnumeration: while ( pellet = [pelletEn nextObject] ) // label for pellet enumeration, used below
	{
		AABB * pelletAABB = [pellet worldAABB];
		
		if ( [camAABB intersects:pelletAABB] )
		{			
			[self removePellet:pellet];
			score += PELLET_POINT;
			continue;
		}
		
		enemyEn = [eatingEnemies objectEnumerator];
		while ( enemy = [enemyEn nextObject] )
		{
			AABB * enemyAABB = [enemy worldAABB];
			
			if ( [enemyAABB intersects:[pellet worldAABB]] )
			{
				[self removePellet:pellet];
				goto pelletEnumeration; // I hate goto statements but I need to break out of 2 control loops
			}
		}
	}
}

- (void) collideWithPowerUps
{
	NSEnumerator * en = [powerUps objectEnumerator];
	SBSceneEntity * powerUp;
	AABB * camAABB = [camera worldAABB];
	while ( powerUp = [en nextObject] )
	{
		if ( [camAABB intersects:[powerUp worldAABB]] )
		{
			// rather than creating a new class to store our type of
			//  - powerup lets just use the scene entity name - it will do
			NSString * name = [powerUp name];
			
			if ( [name isEqualToString:@"retreatingPowerUp"] )
			{
				[self setEnemyStates:ES_RETREATING];
			}
			else if ( [name isEqualToString:@"elevatedPowerUp"] )
			{
				[self setEnemyStates:ES_ELEVATED];
			}
			else if ( [name isEqualToString:@"bouncingPowerUp"] )
			{
				[self setEnemyStates:ES_BOUNCING];
			}
			else if ( [name isEqualToString:@"eatingPowerUp"] )
			{
				[self setEnemyStates:ES_EATING];
			}

			[self removePowerUp:powerUp];
		}
	}
}

#pragma mark ---- Path Points for AI path finding ----

- (void)setPathPoints:(SBPathPoints *)_pathPoints
{
	if ( pathPoints )
	{
		[pathPoints release];
	}

	pathPoints = [_pathPoints retain];
}

- (SBPathPoints *)pathPoints { return pathPoints; }

- (void) setAABB:(AABB *)_aabb
{
	if ( aabb )
	{
		[aabb release];
	}
	aabb = [_aabb retain];
}

#pragma mark ---- Camera ----

- (void) setPlayerStartLoc:(SBVector3D *)_startLoc
{
	[playerStartLoc assign:_startLoc];
}

- (void)setCamera:(SBCamera *)_camera { camera = _camera; }
- (SBCamera *)camera { return camera; }

#pragma mark ---- Octree ----

- (void) addToOctree:(SBSceneEntity *)entity
{
	// only try to add this object if it has drawables
	if ( [[entity drawables] count] > 0 )
	{
		[rootOctreeNode addObject:entity bestFit:YES];
	}

	// now add all the children of this entity to the octree
	NSEnumerator * en = [[entity children] objectEnumerator];
	SBSceneEntity * child;
	while ( child = [en nextObject] )
	{
		[self addToOctree:child];
	}
}

- (void) calculateOctree
{	
	// if we already have an octree, lets clean it up
	if ( rootOctreeNode )
	{
		[rootOctreeNode release];
		
		// reset the entity octNode pointers
		NSEnumerator * en = [sceneEntities objectEnumerator];
		SBSceneEntity * entity;
		while ( entity = [en nextObject] )
		{
			[entity clearOctNodes];
		}
	}
	
	// create the tree
	rootOctreeNode = [[SBOctreeNode alloc] initWithParent:nil extents:aabb depth:0];

	// add the objects
	NSEnumerator * en = [sceneEntities objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[self addToOctree:entity];
	}
}

- (SBOctreeNode *)rootOctreeNode { return rootOctreeNode; }

#pragma mark ---- Scene Assets ----

- (void)setSize:(unsigned int)_size { size = _size; }
- (unsigned int)size { return size; }
- (int) score { return score; }
- (int) remainingLives { return remainingLives; }

- (void) setupLights
{
	strobeRed = strobeGreen = strobeBlue = lightPulseUp = strobe = NO;
	
	SBLight * light = [[SBLight alloc] init];
	[light setLightNum:GL_LIGHT0];
	[lights addObject:light];
	
	[self resetLights];
}

- (void) resetLights
{
	strobeRed = strobeGreen = strobeBlue = lightPulseUp = strobe = NO;

	SBLight * light = [lights objectAtIndex:0];
	
	[[light position] setWithValuesX:20.0 y:2.0 z:1.0];
	[[light ambient] setWithValuesRed:0.0 green:0.0 blue:0.0];
	[[light diffuse] setWithValuesRed:1.0 green:1.0 blue:1.0];
	[[light specular] setWithValuesRed:1.0 green:1.0 blue:1.0];
}

- (void) updateLights
{
	// implement a basic strobe effect for when the enemies are eatable
	if ( strobe )
	{
		SBVectorRGB * diff = [[lights objectAtIndex:0] diffuse];
		SBVectorRGB * spec = [[lights objectAtIndex:0] specular];
		
		float strobeVal;
		
		if ( strobeRed )
		{
			strobeVal = [diff red];
		}
		else if ( strobeGreen )
		{
			strobeVal = [diff green];
		}
		else
		{
			strobeVal = [diff blue];
		}
		
		if ( strobeVal <= 0.1 )
		{
			lightPulseUp = YES;
		}
		else if ( strobeVal >= 1.0 )
		{
			lightPulseUp = NO;
		}
		
		strobeVal = lightPulseUp ? strobeVal + 0.1 : strobeVal - 0.1;
		
		if ( strobeRed )
		{
			[diff setRed:strobeVal];
			[spec setRed:strobeVal];
		}
		if ( strobeGreen )
		{
			[diff setGreen:strobeVal];
			[spec setGreen:strobeVal];
		}
		if ( strobeBlue )
		{
			[diff setBlue:strobeVal];
			[spec setBlue:strobeVal];
		}
	}
}

- (NSArray *)lights	{	return lights;	}

- (void)loadSounds
{
	playerDieSnd = [[NSSound soundNamed:@"player_die"] retain];
	eatGhostSnd = [[NSSound soundNamed:@"eat_ghost"] retain];
	ghostStartsSnd = [[NSSound soundNamed:@"ghost_starts"] retain];
	eatPelletBSnd = [[NSSound soundNamed:@"eat_pelletB"] retain];
	eatPelletB2Snd = [[NSSound soundNamed:@"eat_pelletB2"] retain];
	powerUpSnd = [[NSSound soundNamed:@"eat_superpellet"] retain];
	levelClearSnd = [[NSSound soundNamed:@"level_clear"] retain];
	gameOverSnd = [[NSSound soundNamed:@"gameover"] retain];
}

- (void) loadTextures
{
	//NSString * bundlePath = [[NSBundle mainBundle] resourcePath];
	//NSString * texturePath = [NSString stringWithFormat:@"%@/%s", bundlePath, "pac.tga"];
	
	//[Controller loadGLTexture:texturePath];
}

#pragma mark ---- Debug ----

- (void) logOctree { [rootOctreeNode logValues]; }

- (void) logSceneEntities
{
	NSEnumerator * en = [sceneEntities objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity logValues];
	}
	NSLog( @" " );
}

@end