//
//  SBSceneEntity.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBSceneEntity.h"
#import "SBDrawable.h"
#import "SBMat4.h"
#import "AABB.h"
#import "SBVector3D.h"

@implementation SBSceneEntity

- (id)init
{
	if ( self = [super init] )
	{
		drawables = [[NSMutableArray alloc] init];
		useAlternateTexture = NO;
		
		children = [[NSMutableArray alloc] init];
		parent = nil;
		
		localM = [[SBMat4 alloc] initWithIdentity];
		worldM = [[SBMat4 alloc] initWithIdentity];
		
		localAABB = [[AABB alloc] init];
		worldAABB = [[AABB alloc] init];
		
		octNode = nil;
		collidable = YES;
		
		name = [[NSString alloc] init];
		
		return self;
	}
	return nil;
}

//- (id)initWithSceneEntity:(SBSceneEntity *)_sceneEntity deepCopy:(BOOL)_deep
- (id)initWithSceneEntity:(SBSceneEntity *)_sceneEntity
{
	if ( self = [super init] )
	{
		// create a new array of pointers to drawables rather than retaining the current dictionary
		//  - this way we can remove drawables if we decide from only this object
		drawables = [[NSMutableArray alloc] initWithArray:[_sceneEntity drawables] copyItems:NO];
		useAlternateTexture = [_sceneEntity useAlternateTexture];
		
		// likewise for children entities
		children = [[NSMutableArray alloc] initWithArray:[_sceneEntity children] copyItems:NO];
		
		parent = [_sceneEntity parent];

		// do a deep copy on the transform so we can move this object elsewhere
		localM = [[[_sceneEntity localTransform] copy] retain];
		worldM = [[[_sceneEntity worldTransform] copy] retain];

		localAABB = [[AABB alloc] initWithAABB:[_sceneEntity localAABB]];
		worldAABB = [[AABB alloc] initWithAABB:[_sceneEntity worldAABB]];

		octNode = [_sceneEntity octNode];
		collidable = [_sceneEntity collidable];

		name = [[_sceneEntity name] copy];

		return self;
	}
	return nil;
}

- (id)copyWithZone:(NSZone *)zone
{
	return [[[[self class] allocWithZone: zone] initWithSceneEntity:self] autorelease];
}

/*- (id)deepCopy
{
	return [[[[self class] allocWithZone: nil ] initWithSceneEntity:self deepCopy:YES] autorelease];
}*/

- (void) dealloc
{
	[drawables release];
	[children release];
	[localM release];
	[worldM release];
	[localAABB release];
	[worldAABB release];
	
	[super dealloc];
}

- (void) addDrawable: (SBDrawable *)newDrawable
{
	if ( newDrawable != nil )
	{
		[drawables addObject:newDrawable];
	}
}

- (void) setUseAlternateTexture:(BOOL)_alt
{
	useAlternateTexture = _alt;
}
- (BOOL)useAlternateTexture { return useAlternateTexture; }

- (void) addChild: (SBSceneEntity *) newChild
{
	if ( newChild != nil )
	{
		[newChild setParent:self];
		[children addObject:newChild];
	}
}

- (void) setParent: (SBSceneEntity *) newParent
{
	parent = newParent;
}


- (void) setLocalTransform: (SBMat4 *)newTransform
{
	[localM assign:newTransform];
}

- (void) setLocalAABB: (AABB *)newAABB
{
	[localAABB assign:newAABB];
}

- (void) setOctNode:(SBOctreeNode *)_octNode
{
	octNode = _octNode;
}

- (void) setCollidable:(BOOL)_collidable
{
	collidable = _collidable;
}
- (BOOL)collidable { return collidable; }

// recursively sets the octNode to nil
- (void) clearOctNodes
{
	octNode = nil;
	
	// clear the children
	NSEnumerator * en = [children objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity updateWorldTransform];
	}
}

- (void) setName: (NSString *) newName
{
	[name release];
	name = [newName retain];
}

- (void) updateWorldTransform
{
	// if we have a parent, concat the local transform with its transform
	if ( parent )
	{
		[worldM assign:[[parent localTransform] multiplyByMat4:localM]];
	}
	else
	{
		// else just use the local as the world
		[worldM assign:localM];
	}
	
	// recalculate the world AABB as the transform may have changed
	[self calculateWorldAABB];
	
	// update all the children entities
	NSEnumerator * en = [children objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity updateWorldTransform];
	}
}

- (void) calculateWorldAABB
{
	[worldAABB assign:localAABB];
	[worldAABB transform:worldM];
}

// rely's on worldAABBs already being calculated
- (void) calculateAABBFromChildren:(AABB *)aabb
{
	// include the children
	NSEnumerator * en = [children objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		[entity calculateAABBFromChildren:aabb];
	}
	
	// now include ours
	if ( ![[worldAABB min] isMax] && ![[worldAABB max] isMin] )
	{
		[aabb includeAABB:worldAABB];
	}
}

- (BOOL) recursiveRemoveEntity:(SBSceneEntity *)_entity
{
	// check the children of this node to see if they are the entity we want to remove
	unsigned int idx = [children indexOfObject:_entity];
	if ( idx == NSNotFound )
	{
		// now recurse
		NSEnumerator * en = [children objectEnumerator];
		SBSceneEntity * entity;
		while ( entity = [en nextObject] )
		{
			if ( [entity recursiveRemoveEntity:_entity] )
			{
				return YES;
			}
		}
	}
	else
	{
		[children removeObjectAtIndex:idx];
		return YES;
	}
	
	return NO;
}

// getters
- (NSArray *) drawables { return drawables; }
- (NSArray *) children { return children; }
- (SBSceneEntity *) parent { return parent; }
- (SBMat4 *) localTransform { return localM; }
- (SBMat4 *) worldTransform { return worldM; }
- (AABB *) localAABB { return localAABB; }
- (AABB *) worldAABB { return worldAABB; }
- (SBOctreeNode *) octNode { return octNode; }
- (NSString *) name { return name; }

- (void) updateWithTime:(float)time
{
	// iterate through controllers here and update the scene entity
	
	// test stuff
	//NSLog( @"Children: %d", [children count] );
	/*if ( [name isEqualToString:@"floor"] )
	{
		SBVector3D * temp = [[localM getTranslation] retain];
		[temp setY:[temp y]+0.01];
		[localM setTranslation:temp];
		[temp release];
	}*/
	
	NSEnumerator * en = [children objectEnumerator];
	id entity;
	while ( entity = [en nextObject] )
	{
		[entity updateWithTime:time];
	}
}

- (SBSceneEntity *) getSceneEntityWithName:(NSString *)entityName
{
	NSEnumerator * en = [children objectEnumerator];
	SBSceneEntity * entity;
	while ( entity = [en nextObject] )
	{
		if ( [[entity name] isEqualToString:entityName] )
		{
			return entity;
		}
		
		// check this entities children
		SBSceneEntity * child = [entity getSceneEntityWithName:entityName];
		if ( child )
		{
			return child;
		}
	}
	return nil;
}

- (void) logValues
{
	NSLog( @"Scene Entity: %@\n - drawables: %d\n - children: %d\n - local transform...",
		   name, [drawables count], [children count] );
	[localM logValues];
	NSLog( @" - world transform..." );
	[worldM logValues];
}

@end


