//
//  SBVector3D.m
//  SBGame
//
//  Created by Stuart Bryson on August 2005.
//  Copyright 2005 Stuart Bryson. All rights reserved.
//

#import "SBVector3D.h"


@implementation SBVector3D
// Class methods
+ (SBVector3D *)vector3DWithValuesX:(float)newX y:(float)newY z:(float)newZ
{	return [[[self alloc] initWithValuesX:newX y:newY z:newZ] autorelease];		}

+ (SBVector3D *)vector3DWithArray:(NSArray *)array
{	return [[[self alloc] initWithArray:array] autorelease];	}

+ (SBVector3D *)minVector
{	return [[[self alloc] initWithMin] autorelease];	}
+ (SBVector3D *)maxVector
{	return [[[self alloc] initWithMax] autorelease];	}

- (id) copyWithZone:(NSZone *)zone
{
    return [[[SBVector3D allocWithZone:zone] initWithVector3D:self] autorelease];
}

// Initialisation with values for the components
- (id)initWithValuesX:(float)newX y:(float)newY z:(float)newZ
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		x = newX;
		y = newY;
		z = newZ;
		return self;
	}
	return nil; // something went wrong.
}

// Initialisation with another vector
- (id)initWithVector3D:(SBVector3D *)newVector;
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		x = [newVector x];
		y = [newVector y];
		z = [newVector z];
		return self;
	}
	return nil; // something went wrong.
}

// Super init is called within initWithValues
- (id)initWithArray:(NSArray *)array
{
	return [self initWithValuesX:[[array objectAtIndex:0] floatValue]
			y:[[array objectAtIndex:1] floatValue]
			z:[[array objectAtIndex:2] floatValue]];
}

- (id)initWithMax
{
	return [self initWithValuesX:FLT_MAX y:FLT_MAX z:FLT_MAX];
}

- (id)initWithMin
{
	return [self initWithValuesX:-FLT_MAX y:-FLT_MAX z:-FLT_MAX];
}

// Assignment method
- (void)assign:(SBVector3D *)rhs
{
	x = [rhs x];
	y = [rhs y];
	z = [rhs z];
}

- (NSArray *)asArray
{	return [NSArray arrayWithObjects:[NSNumber numberWithFloat:x], [NSNumber numberWithFloat:y], [NSNumber numberWithFloat:z], nil];	}

// *** Setter methods ***
- (void)setWithValuesX:(float)newX y:(float)newY z:(float)newZ
{
	x = newX;
	y = newY;
	z = newZ;
}

- (void)setX:(float)newX	{	x = newX;	}
- (void)setY:(float)newY	{	y = newY;	}
- (void)setZ:(float)newZ	{	z = newZ;	}

- (void)setMin { x = -FLT_MAX; y = -FLT_MAX; z = -FLT_MAX; }
- (void)setMax { x = FLT_MAX; y = FLT_MAX; z = FLT_MAX; }

// *** Getter methods ***
- (float)x					{	return x;	}
- (float)y					{	return y;	}
- (float)z					{	return z;	}

// Normalise the current vector
- (void)normalise
{
	float length = [self length];
	x /= length;
	y /= length;
	z /= length;
}

// Returns the unit vector of the current object (ie divided by its own length)
- (SBVector3D *)unit	{	return [self divideByFloat:[self length]];	}

// Returns the length
- (float)length		{	return sqrt(x * x + y * y + z * z);			}

- (float)distanceTo:(SBVector3D *)rhs
{	return [[self subtractVector:rhs] length];	}

// Adds another vector to the vector
- (SBVector3D *)addVector:(SBVector3D *)rhs
{	return [SBVector3D vector3DWithValuesX:(x + [rhs x]) y:(y + [rhs y]) z:(z + [rhs z])];	}

// Subtract another vector from the vector
- (SBVector3D *)subtractVector:(SBVector3D *)rhs
{	return [SBVector3D vector3DWithValuesX:(x - [rhs x]) y:(y - [rhs y]) z:(z - [rhs z])];	}

// Multiply the vector by a float
- (SBVector3D *)multiplyByFloat:(float)rhs
{	return [SBVector3D vector3DWithValuesX:(x * rhs) y:(y * rhs) z:(z * rhs)];				}

// Mulitpy the vector by another vector
- (SBVector3D *)multiplyByVector:(SBVector3D *)rhs
{	return [SBVector3D vector3DWithValuesX:(x * [rhs x]) y:(y * [rhs y]) z:(z * [rhs z])];	}

// Divides the vector by a float
- (SBVector3D *)divideByFloat:(float)rhs
{	return [SBVector3D vector3DWithValuesX:(x / rhs) y:(y / rhs) z:(z / rhs)];				}

// Divides the vector by another Vector
- (SBVector3D *)divideByVector:(SBVector3D *)rhs
{	return [SBVector3D vector3DWithValuesX:(x / [rhs x]) y:(y / [rhs y]) z:(z / [rhs z])];	}

- (BOOL)isZero
{
	float eps = 0.0001;
	return x <  eps &&  y <  eps && z <  eps;
}

- (BOOL)isMin
{
	return ( x == -FLT_MAX && x == -FLT_MAX && x == -FLT_MAX );
}

- (BOOL)isMax
{
	return ( x == FLT_MAX && x == FLT_MAX && x == FLT_MAX );
}

- (BOOL)equals:(SBVector3D *)rhs
{
	float eps = 0.0001;
	return fabsf(x - [rhs x]) <  eps &&  fabsf(y - [rhs y]) <  eps && fabsf(z - [rhs z]) <  eps;
}

// Dot product of two vectors
- (float)dotProduct:(SBVector3D *)rhs		{	return (x * [rhs x] + y * [rhs y] + z * [rhs z]);	}

// Cross product of two vectors
- (SBVector3D *)crossProduct:(SBVector3D *)rhs
{	return [SBVector3D vector3DWithValuesX:(y * [rhs z] - z * [rhs y]) y:(z * [rhs x] - x * [rhs z]) z:(x * [rhs y] - y * [rhs x])];	}

// Logs the vector Values
- (void)logValues	{	NSLog(@"X = %f, Y = %f, Z = %f", x, y, z);	}
@end
