//
//  Camera.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Sat Aug 11 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Camera.h"


@implementation Camera
// Class methods
+ (Camera *)cameraWithDictionary:(NSDictionary *)dictionary
{	return [[[Camera alloc] initWithDictionary:dictionary] autorelease];	}

// This will initialise the camera with the appropriate data as to the current frame in a linear fashion
+ (Camera *)cameraWithLinearFrameStart:(Camera *)start end:(Camera *)end currentFrame:(int)currentFrame totalFrames:(int)numFrames
{
	return [[[Camera alloc] initWithValuesA: [start a] + (([end a] - [start a])/(numFrames - 1))*currentFrame
							b:[start b] + (([end b] - [start b])/(numFrames - 1))*currentFrame
							c:[start c] + (([end c] - [start c])/(numFrames - 1))*currentFrame
							d:[start d] + (([end d] - [start d])/(numFrames - 1))*currentFrame
							e:[start e] + (([end e] - [start e])/(numFrames - 1))*currentFrame
							f:[start f] + (([end f] - [start f])/(numFrames - 1))*currentFrame
							g:[start g] + (([end g] - [start g])/(numFrames - 1))*currentFrame
							twist:[start twist] + (([end twist] - [start twist])/(numFrames - 1))*currentFrame
							] autorelease];
}

+ (Camera *)cameraWithBezierFrames:(NSArray *)controlPoints currentFrame:(int)currentFrame totalFrames:(int)totalFrames
{
	
	int c[[controlPoints count]];
	int i, k, n = [controlPoints count]-1;
	float blend, u = currentFrame / (totalFrames > 0 ? totalFrames : 1);
	Camera *camera = [[[Camera alloc] init] autorelease];
	Vector3D *newLocation = [[Vector3D alloc] init];
	Vector3D *newLookAt = [[Vector3D alloc] init];
	
	for (k=0; k <= totalFrames-1; k++)
	{
		c[k] = 1;
		for (i = totalFrames-1; i >= k+1; i--)
			c[k] *= i;
		for (i = totalFrames-1-k; i >= 2; i--)
			c[k] /= i;
	}
	
	for (k=0; k<[controlPoints count]; k++)
	{
		Camera *currentControl = [controlPoints objectAtIndex:k];
		blend = c[k] * pow(u, k) * pow(1-u, n-k);
		[newLocation assign:[newLocation addVector:[[currentControl location] multiplyByFloat:blend]]];
		[newLookAt assign:[newLookAt addVector:[[currentControl lookAt] multiplyByFloat:blend]]];
	}
	
	[camera setLocation:newLocation];
	[camera setLookAt:newLookAt];
	[camera setD:[[controlPoints objectAtIndex:0] d]];
	[camera setTwist:[[controlPoints objectAtIndex:0] twist]];

	[newLocation release];
	[newLookAt release];

	return camera;
}

// Standard initialisation
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] init];
		lookAt = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// Initialise with lots of individual values. This should never be used
- (id)initWithValuesA:(float)newA b:(float)newB c:(float)newC d:(float)newD e:(float)newE f:(float)newF g:(float)newG twist:(float)newTwist
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] initWithValuesX:newA y:newB z:newC];
		lookAt = [[Vector3D alloc] initWithValuesX:newE y:newF z:newG];
		d = newD;
		twist = newTwist;
		return self;
	}
	return nil; // something went wrong.
}

// Prefered initialisation method, much more compact
- (id)initWithVectorsLocation:(Vector3D *)newLocation d:(float)newD lookAt:(Vector3D *)newLookAt twist:(float)newTwist
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [newLocation retain];
		lookAt = [newLookAt retain];
		d = newD;
		twist = newTwist;
		return self;
	}
	return nil; // something went wrong.
}

- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Location"]];
		lookAt = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Look At"]];
		d = [[dictionary valueForKey:@"View Plane Distance"] floatValue];
		twist = [[dictionary valueForKey:@"Twist"] floatValue];
		return self;
	}
	return nil; // something went wrong.
}

- (NSDictionary *)asDictionary
{
	return [NSDictionary dictionaryWithObjects:[NSArray arrayWithObjects:
				[location asArray],
				[lookAt asArray],
				[NSNumber numberWithFloat:d],
				[NSNumber numberWithFloat:twist],
				nil]
				// give the above objects keys in the dictionary
				forKeys:[NSArray arrayWithObjects:@"Location", @"Look At", @"View Plane Distance", @"Twist", nil]
			];
}

// Assignment method
- (void)assign:(Camera *)rhs
{
	[location assign:[rhs location]];
	[lookAt assign:[rhs lookAt]];
	d = [rhs d];
	twist = [rhs twist];
}

// Copy constructor
- (id)copyWithZone:(NSZone *)zone
{	return [[Camera allocWithZone:zone] initWithVectorsLocation:location d:d lookAt:lookAt twist:twist];	}

// Destructor
- (void)dealloc
{
	[location release];
	[lookAt release];
	[super dealloc];  //use the super class's method for instance destruction
}

// Big setters
- (void)setWithValuesA:(float)newA b:(float)newB c:(float)newC d:(float)newD e:(float)newE f:(float)newF g:(float)newG twist:(float)newTwist
{
	[location setX:newA];
	[location setY:newB];
	[location setZ:newC];
	d = newD;
	[lookAt setX:newE];
	[lookAt setY:newF];
	[lookAt setZ:newG];
	twist = newTwist;
}

// Prefered big setter, far less ugly
- (void)setWithVectorsLocation:(Vector3D *)newLocation d:(float)newD lookAt:(Vector3D *)newLookAt twist:(float)newTwist
{
	[location assign:newLocation];
	[lookAt assign:newLookAt];
	d = newD;
	twist = newTwist;
}

// Setters
- (void)setA:(float)newA { [location setX:newA]; }
- (void)setB:(float)newB { [location setY:newB]; }
- (void)setC:(float)newC { [location setZ:newC]; }
- (void)setD:(float)newD { d = newD; }
- (void)setE:(float)newE { [lookAt setX:newE]; }
- (void)setF:(float)newF { [lookAt setY:newF]; }
- (void)setG:(float)newG { [lookAt setZ:newG]; }
- (void)setTwist:(float)newTwist { twist = newTwist; }
- (void)setLocation:(Vector3D *)newLocation	{	[location assign:newLocation];	}
- (void)setLookAt:(Vector3D *)newLookAt		{	[lookAt assign:newLookAt];		}

// Getters, you see how these return part of the location and lookAt Vector3Ds. This is for RenderScene
- (float)a { return [location x]; }
- (float)b { return [location y]; }
- (float)c { return [location z]; }
- (float)d { return d; }
- (float)e { return [lookAt x]; }
- (float)f { return [lookAt y]; }
- (float)g { return [lookAt z]; }
- (float)twist { return twist; }

- (Vector3D *)location	{	return location;	}
- (Vector3D *)lookAt	{	return lookAt;		}

// No need to store this, there can be only one camera, at least for now... <maniacal laugh>
- (NSString *)name		{ return @"Camera";	}

@end
