//
//  Checkerboard.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Sep 28 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Checkerboard.h"


@implementation Checkerboard

// Class methods
+ (Checkerboard *)checkerboardWithDictionary:(NSDictionary *)dictionary
{	return [[[Checkerboard alloc] initWithDictionary:dictionary] autorelease];		}

// Normal constructor, probably never called, but what the hey
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
//		primary = [[Material alloc] init];
//		secondary = [[Material alloc] init];
		size = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// Used when we are loading a scene file
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		int i;
		id mat;

		primary = secondary = nil;
		//For every object in our threeDObjects array
		for (i=0; i<[[dictionary valueForKey:@"Materials"] count]; i++)
		{
			mat = [[dictionary valueForKey:@"Materials"] objectAtIndex:i];
			if ([mat isKindOfClass:[Material class]])
			{
				if ([[mat name] isEqualToString:[dictionary valueForKey:@"Primary Check"]])
					primary = mat;
				if ([[mat name] isEqualToString:[dictionary valueForKey:@"Secondary Check"]])
					secondary = mat;
			}
		}
		size = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Size"]];
		return self;
	}
	return nil; // something went wrong.
}

- (id)copyWithZone:(NSZone *)zone	{	return [[Checkerboard allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

// Destructor
- (void)dealloc
{
//	[primary release];
//	[secondary release];
	[size release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// Used when we are saving to a scene file
- (NSMutableDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[primary name] forKey:@"Primary Check"];
	[dictionary setObject:[secondary name] forKey:@"Secondary Check"];
	[dictionary setObject:[size asArray] forKey:@"Size"];
	[dictionary setObject:@"Checkerboard" forKey:@"Material Type"];
	return dictionary;
}

// Assignment method
- (void)assign:(Checkerboard *)rhs
{
	[super assign:rhs];
	primary = [rhs primary];
	secondary = [rhs secondary];
	[size assign:[rhs size]];
}

// Getters
- (Material *)primary		{	return primary;		}
- (Material *)secondary		{	return secondary;	}
- (Vector3D *)size			{	return size;		}

// Setters
- (void)setPrimary:(Material *)newMaterial		{	primary = newMaterial;		}
- (void)setSecondary:(Material *)newMaterial	{	secondary = newMaterial;	}
- (void)setSize:(Vector3D *)newSize				{	[size assign:newSize];		}

// Diffuse reflection at a point, used for textures
- (float)diffuseReflection:(Vector3D *)hitPoint		{	return [[self whichCheck:hitPoint] diffuseReflection:hitPoint];		}
// Specular reflection at a point, used for textures
- (float)specularReflection:(Vector3D *)hitPoint	{	return [[self whichCheck:hitPoint] specularReflection:hitPoint];	}
// Global reflection, how much of the world can you see reflected in this object
- (float)globalReflection:(Vector3D *)hitPoint		{	return [[self whichCheck:hitPoint] globalReflection:hitPoint];		}
// The amount of light which gets transmitted through an object
- (float)globalTransmission:(Vector3D *)hitPoint	{	return [[self whichCheck:hitPoint] globalTransmission:hitPoint];	}
// Size of specular reflection
- (int)specularExponent:(Vector3D *)hitPoint		{	return [[self whichCheck:hitPoint] specularExponent:hitPoint];		}
// If the object is diffuse reflective at a point, ie diffuseReflection is greater than zero
- (BOOL)isDiffuseReflective:(Vector3D *)hitPoint	{	return [[self whichCheck:hitPoint] isDiffuseReflective:hitPoint];	}
// If the object is specular reflective at a point, ie specularReflection is greater than zero
- (BOOL)isSpecularReflective:(Vector3D *)hitPoint	{	return [[self whichCheck:hitPoint] isSpecularReflective:hitPoint];	}
// Combination of the two above
- (BOOL)isReflective:(Vector3D *)hitPoint			{	return ([self isDiffuseReflective:hitPoint] || [self isSpecularReflective:hitPoint]);	}
// It the object transparent at this point
- (BOOL)isTransparent:(Vector3D *)hitPoint			{	return [[self whichCheck:hitPoint] isTransparent:hitPoint];	}
// The colour of the object.
- (RGBVector *)diffuseColour:(Vector3D *)hitPoint	{	return [[self whichCheck:hitPoint] diffuseColour:hitPoint];	}

// This is from Hill (2000)
- (Material *)whichCheck:(Vector3D *)hitPoint
{
	int offset = 100;
	int jump = ((int)(offset + [hitPoint x] / [size x]) + (int)(offset + [hitPoint y] / [size y]) + (int)(offset + [hitPoint z] / [size z])) % 2;

	if (jump == 0)
		return primary;
	else
		return secondary;
}

@end
