//
//  Controller.m
//  Controller
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Controller.h"

#define SAMPLE_SIZE 8
#define SAMPLE_NUMBER 3
#define PIXEL_BYTES 3

#define PREVIEW 0
#define SIGNATURE 1
#define RENDER 2

@implementation Controller

- (id)init
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		NSNotificationCenter *nc;
		nc = [NSNotificationCenter defaultCenter];
		[nc addObserver: self selector: @selector(updateProgress:) name: @"updateProgress" object: nil];
		[nc addObserver: self selector: @selector(drawNotification:) name: @"drawNotification" object: nil];
		[nc addObserver: self selector: @selector(findObject:) name: @"findObject" object:nil];
		loadedPreferences = [[Preferences alloc] init];
		viewRect = [rayImageViewer bounds];
		theWorld = [[World alloc] initWithPreferences:loadedPreferences];
		sceneLoaded = NO;
		sceneName = [[NSMutableString alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

- (void)dealloc
{
	free(bitmapData);
	[theWorld release];
	[loadedPreferences release];
	[theImage release];
	[theBitmapRep release];
	
	{
	NSNotificationCenter *nc;
	nc = [NSNotificationCenter defaultCenter];
    [nc removeObserver: self];
	}
	
	[super dealloc];  //use the super class's method for instance destruction
}

- (IBAction)openScene:(id)sender
{
	BOOL closed = YES;
	if (sceneLoaded)
		closed = [self actuallyCloseScene];
	
	if (closed)
	{
		NSOpenPanel *oPanel = [NSOpenPanel openPanel];	
		// Display standard open file panel
		[oPanel setAllowsMultipleSelection:NO];
		if ([oPanel runModalForDirectory:@"/Users/spbryson/Documents/RayTracerDocs/Scenes" file:nil types:[NSArray arrayWithObject:@"plist"]] == NSOKButton)
		{
			NSArray *filesToOpen = [oPanel filenames];
			[sceneName setString:[filesToOpen objectAtIndex:0]];
		}
		if ([theWorld BuildWithName:sceneName])
		{
			[object3DView reloadData];
			[object3DView setDoubleAction:@selector(setObjectBeingEdited:)];
			[bRun setEnabled:YES];
			sceneLoaded = YES;
		}
	}
}

- (IBAction)importMaterialLibrary:(id)sender
{
    if (sceneLoaded)
    {
        NSOpenPanel *oPanel = [NSOpenPanel openPanel];	
        // Display standard open file panel
        [oPanel setAllowsMultipleSelection:NO];
        if ([oPanel runModalForDirectory:@"/Users/spbryson/Documents/RayTracerDocs/Scenes" file:nil types:[NSArray arrayWithObject:@"plist"]] == NSOKButton)
        {
            NSArray *filesToOpen = [oPanel filenames];
            [sceneName setString:[filesToOpen objectAtIndex:0]];
        }
        if ([theWorld buildMaterials:sceneName])
        {
            [object3DView reloadData];
            [object3DView setDoubleAction:@selector(setObjectBeingEdited:)];
            [bRun setEnabled:YES];
            sceneLoaded = YES;
        }        
    }
}

- (IBAction)closeScene:(id)sender
{
	[self actuallyCloseScene];
}

- (IBAction)newScene:(id)sender
{
	if ([theWorld BuildWithName:nil])
	{
		[sceneName setString:@""];
		[object3DView reloadData];
		[object3DView setDoubleAction:@selector(setObjectBeingEdited:)];
		[bRun setEnabled:YES];
		sceneLoaded = YES;
	}
}

- (IBAction)saveScene:(id)sender
{
	int status;
	BOOL saved;
	if ([sceneName isEqualToString:@""])
		[self saveSceneAs:self];
	else
	{
	saved = [theWorld saveSceneWithName:sceneName];
	if (!saved && (status = NSRunAlertPanel(@"Warning", @"Could not save the scene. Try again?", @"Yes", @"Cancel", @"No", NULL)) == NSAlertDefaultReturn)
		[self saveScene:self];
	}
}

- (IBAction)saveSceneAs:(id)sender
{
	BOOL saved;
	int status;
	// Straight from Apple documentation
	NSSavePanel *sp;
	/* create or get the shared instance of NSSavePanel */
	sp = [NSSavePanel savePanel];
	/* set up new attributes for save */
	[sp setRequiredFileType:@"plist"];
	/* if successful, save file under designated name */
	if ([sp runModalForDirectory:@"/Users/spbryson/Documents/RayTracerDocs/Scenes" file:@""] == NSOKButton)
	{
		[sceneName setString:[sp filename]];
		saved = [theWorld saveSceneWithName:sceneName];
		if (!saved && (status = NSRunAlertPanel(@"Warning", @"Could not save the scene. Try again?", @"Yes", @"Cancel", @"No", NULL)) == NSAlertDefaultReturn)
			[self saveScene:self];
	}
}

- (BOOL)actuallyCloseScene
{
	// Set it to yes so that if they chose not to save it, it won't give a warning below.
	BOOL saved = YES;
	int status;
	// Create alert panel
    if ((status = NSRunAlertPanel(@"Warning", @"Do you wish to save the current scene?", @"Yes", @"Cancel", @"No", NULL)) == NSAlertDefaultReturn)
		saved = [theWorld saveSceneWithName:sceneName];
	// If they cancel
	if (status == NSAlertAlternateReturn)
		return NO;
	// If (they do not require to save the scene) OR (the scene was saved) OR
	// (they tried to save it but it didn't save and they acknowledged the warning but still went ahead with the close)
	if ((status == NSAlertOtherReturn) || (saved == YES) ||
		(!saved && (status = NSRunAlertPanel(@"Warning", @"Could not save the scene. Continue close?", @"Yes", @"Cancel", @"No", NULL)) == NSAlertDefaultReturn))
	{	
		[sceneName setString:@""];
		[theWorld Demolish];
		sceneLoaded = NO;
		[bRun setEnabled:NO];
		[object3DDrawer close];
		[mEditScene setTitle:@"Edit Scene"];
		return YES;
	}
	return NO;
}

- (IBAction)editScene:(id)sender
{
	if (sceneLoaded)
	{
		if ([object3DDrawer state] == NSDrawerClosedState || [object3DDrawer state] == NSDrawerClosingState)
		{
			[object3DDrawer open];
			[mEditScene setTitle:@"Hide Editor"];
		}
		else
		{
			[object3DDrawer close];
			[mEditScene setTitle:@"Edit Scene"];
		}
	}
}

- (IBAction)renderImage:(id)sender
{
	if (sceneLoaded)
	{
		NSDate *startRun = [NSDate date];
		NSString *temp;
		//[NSApp beginSheet:renderSheet modalForWindow:mainWindow modalDelegate:self didEndSelector:NULL contextInfo:nil];
		//[renderProgress setIndeterminate:NO];
		//[renderProgress setMaxValue:([loadedPreferences pixelsHigh] - 1)];
		[theWorld setBitmapData:bitmapData];
		[theWorld GenerateScene:[self renderStyle]];
		[self drawImage];
		temp = [NSMutableString stringWithString:@"Last Render Time: "];
		[txtStatusBar setStringValue:[temp stringByAppendingString:[[NSNumber numberWithDouble:(-(double)[startRun timeIntervalSinceNow])] stringValue]]];
		//[renderSheet orderOut:nil];
		//[NSApp endSheet:renderSheet];
	}
	else
		NSLog(@"Need to load scene first.");
}

- (void)updateProgress:(NSNotification *)note
{
	// This stuff is commented out as it proved to slow our ray tracer by 10%.
	//NSDictionary *instructions = [note object];
	//[renderProgress setDoubleValue:[[instructions valueForKey: @"pixels"] doubleValue]];
	//[renderProgress display];
	//[txtStatusBar setIntValue:[[instructions valueForKey: @"pixels"] intValue]];
	//[txtStatusBar display];
	
	//Could probably put this in the instructions dictionary sent from the World but less message passing this way.
	if ([loadedPreferences liveUpdate])
	{
		[self drawImage];
		[rayImageViewer display];
	}
}

- (IBAction)saveImage:(id)sender
{
	NSSavePanel *sp;
	int saveResult;

	/* create or get the shared instance of NSSavePanel */
	sp = [NSSavePanel savePanel];

	/* set up new attributes for save */
	//[sp setAccessoryView:newView];
	[sp setRequiredFileType:@"tiff"];

	/* display the NSSavePanel */
	saveResult = [sp runModalForDirectory:NSHomeDirectory() file:@""];
	
	/* if successful, save file under designated name */
	if (saveResult == NSOKButton)
	{
		/* create the data for a tiff representation */
		NSData *tiffImage = [[[NSData alloc] initWithData:[theImage TIFFRepresentation]] autorelease];
		if (![tiffImage writeToFile:[sp filename] atomically:YES])
		{
			NSBeep();
			NSLog(@"Could not write tiff image to disk");
		}
	}
}

- (IBAction)setRenderStyle:(id)sender
{
    NSString *style = [renderStylePopup titleOfSelectedItem];
	[renderStylePopup setTitle:style];
}

- (int)renderStyle
{
	NSString *style = [renderStylePopup titleOfSelectedItem];
    if ([style isEqualToString:@"Preview"])
        return PREVIEW;
	else if ([style isEqualToString:@"Signature"])
		return SIGNATURE;
	else if ([style isEqualToString:@"Render"])
		return RENDER;
	return -1;
}

- (IBAction)renderMovie:(id)sender
{
	int numberOfFrames = 40, numberOfKeyFrames = 2;
	[theWorld setBitmapData:bitmapData];
	[theWorld renderMovieWithNumberOfKeyFrames:numberOfKeyFrames numberOfFrames:numberOfFrames size:viewRect];	
}

- (void)drawNotification:(NSNotification *)note
{
	NSLog(@"Draw Note called");
	[self drawImage];
}

- (void)findObject:(NSNotification *)note
{
	NSDictionary *instructions = [note object];
	objBeingEdited = [theWorld findObjectAtRow:[[instructions valueForKey: @"row"] intValue] col:[[instructions valueForKey: @"col"] intValue]];
	[self editObject:self];
}

- (void)awakeFromNib
{
	[self initBitmapData];
	[mEditScene setEnabled:NO];
	[mRun setEnabled:NO];
	[bRun setEnabled:NO];
	//Sets the palette to automatically save its position in the user defaults (prefs) whenever it is moved.
	[spherePalette setFrameAutosaveName:@"Sphere Palette"];
	[trianglePalette setFrameAutosaveName:@"Triangle Palette"];
	[planePalette setFrameAutosaveName:@"Plane Palette"];
	[genericSpherePalette setFrameAutosaveName:@"Generic Sphere Palette"];
	[genericBoxPalette setFrameAutosaveName:@"Generic Box Palette"];
	[genericTorusPalette setFrameAutosaveName:@"Generic Torus Palette"];
	[genericTaperedCylinderPalette setFrameAutosaveName:@"Generic Tapered Cylinder Palette"];
	[cameraPalette setFrameAutosaveName:@"Camera Palette"];
	[lightPalette setFrameAutosaveName:@"Light Palette"];
	[materialPalette setFrameAutosaveName:@"Material Palette"];
	[mainWindow setFrameAutosaveName:@"Main Window"];
}

- (void)initBitmapData
{
	unsigned int i, row, col, byteOffset;
	pixelsWide = [loadedPreferences pixelsWide];
	pixelsHigh = [loadedPreferences pixelsHigh];

	free(bitmapData);
	bitmapData = (void*)calloc(pixelsWide * pixelsHigh * PIXEL_BYTES,sizeof(unsigned char));
	
	for (row = 0; row <= (pixelsHigh - 1); row++)
	{
		for (col = 0; col <= (pixelsWide - 1); col++) 
		{
			for (i = 0; i < 3; i++)
			{
				color[i] = 0;
				if (modf(color[i],&intPart) >= 0.5)
					intColor[i] = intPart + 1;
				else
					intColor[i] = intPart;
			}
		byteOffset = PIXEL_BYTES * ((row * pixelsWide) + col);
		for (i = 0; i < 3; i++)
			bitmapData[byteOffset + i] = intColor[i];
		}
	}
	[self drawImage];
}

- (void)drawImage
{
	if (theImage != nil)
	{   
		[theImage release]; 
		theImage = nil;
	}
	theImage = [[NSImage alloc] initWithSize:viewRect.size];
	[theImage setFlipped:NO];

	if (theBitmapRep != nil)
	{
		[theBitmapRep release];
		theBitmapRep = nil;
	}
	theBitmapRep = [NSBitmapImageRep alloc];

	[theBitmapRep initWithBitmapDataPlanes:&bitmapData
		pixelsWide:[loadedPreferences pixelsWide]
		pixelsHigh:[loadedPreferences pixelsHigh] 
		bitsPerSample:SAMPLE_SIZE      
		samplesPerPixel:SAMPLE_NUMBER
		hasAlpha:NO
		isPlanar:NO
		colorSpaceName:NSCalibratedRGBColorSpace
		bytesPerRow:([loadedPreferences pixelsWide] * PIXEL_BYTES)
		bitsPerPixel:0];
	[theImage addRepresentation:theBitmapRep];
	[rayImageViewer setImage: theImage];
}

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)sender
{
	return YES;
}

- (IBAction)quit:(id)sender
{
	[object3DDrawer close];
	[NSApp terminate:self];
}

- (IBAction)savePreferences:(id)sender
{
	[loadedPreferences setPreviewBlockSize:[pTxtBlockSize intValue]];
	[loadedPreferences setPixelsWide:[pTxtWidth intValue]];
	[loadedPreferences setPixelsHigh:[pTxtHeight intValue]];
	[loadedPreferences setUseParallelProjection:[pParallel intValue]];
	[loadedPreferences setRecursionDepth:[pTxtRecursion intValue]];
	[loadedPreferences setLiveUpdate:[pLiveUpdate state]];
	[loadedPreferences setAntiAlias:[pAntiAlias state]];
	[loadedPreferences save];
	[self initBitmapData];
	[pPrefsWindow orderOut:nil];
    [NSApp endSheet:pPrefsWindow];
}

- (IBAction)cancelPreferences:(id)sender
{
	[pPrefsWindow orderOut:nil];
    [NSApp endSheet:pPrefsWindow];
}

- (IBAction)preferences:(id)sender
{
	[pTxtWidth setNextText:pTxtHeight];
	[pTxtHeight setNextText:pTxtWidth];
	[loadedPreferences reload];
	[pTxtWidth setIntValue:[loadedPreferences pixelsWide]];
	[pSliderWidth setIntValue:[loadedPreferences pixelsWide]];
	[pTxtHeight setIntValue:[loadedPreferences pixelsHigh]];
	[pSliderHeight setIntValue:[loadedPreferences pixelsHigh]];
	[pTxtBlockSize setIntValue:[loadedPreferences previewBlockSize]];
	[pSliderBlockSize setIntValue:[loadedPreferences previewBlockSize]];
	[pTxtRecursion setIntValue:[loadedPreferences recursionDepth]];
	[pSliderRecursion setIntValue:[loadedPreferences recursionDepth]];
	[pParallel setIntValue:[loadedPreferences useParallelProjection]];
	if ([loadedPreferences useParallelProjection] == 1)
		[pPerspective setIntValue:0];
	else
		[pPerspective setIntValue:1];
	[pLiveUpdate setState:[loadedPreferences liveUpdate]];
	[pAntiAlias setState:[loadedPreferences antiAlias]];
	[NSApp beginSheet:pPrefsWindow modalForWindow:mainWindow
        modalDelegate:self didEndSelector:NULL contextInfo:nil];
}

- (IBAction)updateWidthControls:(id)sender
{
	if ([sender intValue] < 1)
	{
		[pTxtWidth setIntValue:1];
		[pSliderWidth setIntValue:1];
	}
	else
	{
		[pTxtWidth setIntValue:[sender intValue]];
		[pSliderWidth setIntValue:[sender intValue]];
	}
}

- (IBAction)updateHeightControls:(id)sender
{
	if ([sender intValue] < 1)
	{
		[pTxtHeight setIntValue:1];
		[pSliderHeight setIntValue:1];
	}
	else
	{
		[pTxtHeight setIntValue:[sender intValue]];
		[pSliderHeight setIntValue:[sender intValue]];
	}
}

- (IBAction)updatePreviewBlockControls:(id)sender
{
	if ([sender intValue] < 2)
	{
		[pTxtBlockSize setIntValue:2];
		[pSliderBlockSize setIntValue:2];
	}
	else if ([sender intValue] > 20)
	{
		[pTxtBlockSize setIntValue:20];
		[pSliderBlockSize setIntValue:20];
	}
	else
	{
		[pTxtBlockSize setIntValue:[sender intValue]];
		[pSliderBlockSize setIntValue:[sender intValue]];
	}
}

- (IBAction)updateRecursionControls:(id)sender
{
	if ([sender intValue] > 10)
	{
		[pTxtRecursion setIntValue:10];
		[pSliderRecursion setIntValue:10];
	}
	else if ([sender intValue] < 1)
	{
		[pTxtRecursion setIntValue:1];
		[pSliderRecursion setIntValue:1];
	}
	else
	{
		[pTxtRecursion setIntValue:[sender intValue]];
		[pSliderRecursion setIntValue:[sender intValue]];
	}
}


- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item
{ return (item == nil) ? [[theWorld scene] objectAtIndex:index] : [item objectAtIndex:index]; }

- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item
{ return ([item isKindOfClass:[NSMutableArray class]]) ? YES : NO; }

- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{ return (item == nil) ? [[theWorld scene] count] : ([item count] - 1); }

- (id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
   if (item == nil)
	    return nil;
	else if ([item isKindOfClass:[NSMutableArray class]])
		return [item lastObject];
	else if ([item isKindOfClass:[Object3D class]] || [item isKindOfClass:[Light class]]
				|| [item isKindOfClass:[Camera class]] || [item isKindOfClass:[Material class]])
		return [item name];
	else
	    return item;
}
- (BOOL)outlineView:(NSOutlineView *)outlineView shouldEditTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
	return NO;
}

/* Couldn't get this working for saving an object material pointer
- (id)comboBox:(NSComboBox *)aComboBox objectValueForItemAtIndex:(int)index
{
	if ([[[[theWorld scene] objectAtIndex:2] objectAtIndex:index] isKindOfClass:[Material class]])
		return [[[[theWorld scene] objectAtIndex:2] objectAtIndex:index] name];
	else
		return nil;
}

- (int)numberOfItemsInComboBox:(NSComboBox *)aComboBox
{	return [[[theWorld scene] objectAtIndex:2] count];	}
*/

- (IBAction)deleteObject:(id)sender
{
	objBeingEdited = [object3DView itemAtRow:[object3DView selectedRow]];
	objBeingEdited = [object3DView isExpandable:objBeingEdited] ? nil : objBeingEdited;
	if (objBeingEdited != nil)
	{
		if ([objBeingEdited isKindOfClass:[Object3D class]])
		{
			NSMutableArray *threeDObjects = [[theWorld scene] objectAtIndex:0];
			if ([threeDObjects count] > 2)
				[threeDObjects removeObject:objBeingEdited];
		}
		else if ([objBeingEdited isKindOfClass:[Light class]])
		{
			NSMutableArray *lights = [[theWorld scene] objectAtIndex:1];
			if ([lights count] > 2)
				[lights removeObject:objBeingEdited];
		}
		else if ([objBeingEdited isKindOfClass:[Material class]])
		{
			//We need to delete the Material Object, set all objects that pointed to this material to point to the default
			//Do not allow the default to be deleted
		}
	}
	[object3DView reloadData];
	objBeingEdited = nil;
}

- (IBAction)setObjectBeingEdited:(id)sender
{
	objBeingEdited = [object3DView itemAtRow:[object3DView selectedRow]];
	objBeingEdited = [object3DView isExpandable:objBeingEdited] ? nil : objBeingEdited;
	[self editObject:self];
}

- (IBAction)editObject:(id)sender
{
	if (objBeingEdited != nil)
	{
		if ([objBeingEdited isKindOfClass:[Sphere class]])
		{
			currentSphere = objBeingEdited;
			[self refreshSphere:self];
			[spherePalette makeKeyAndOrderFront:spherePalette];
		}
		else if ([objBeingEdited isKindOfClass:[Plane class]])
		{
			currentPlane = objBeingEdited;
			[self refreshPlane:self];
			[planePalette makeKeyAndOrderFront:planePalette];
		}
		else if ([objBeingEdited isKindOfClass:[Camera class]])
		{
			currentCamera = objBeingEdited;
			[self refreshCamera:self];
			[cameraPalette makeKeyAndOrderFront:cameraPalette];
		}
		else if ([objBeingEdited isKindOfClass:[Light class]])
		{
			currentLight = objBeingEdited;
			[self refreshLight:self];
			[lightPalette makeKeyAndOrderFront:lightPalette];
		}
		else if ([objBeingEdited isKindOfClass:[Triangle class]])
		{
			currentTriangle = objBeingEdited;
			[self refreshTriangle:self];
			[trianglePalette makeKeyAndOrderFront:trianglePalette];
		}
		else if ([objBeingEdited isKindOfClass:[GenericBox class]])
		{
			currentGenericBox = objBeingEdited;
			[self refreshGenericBox:self];
			[genericBoxPalette makeKeyAndOrderFront:genericBoxPalette];
		}
		else if ([objBeingEdited isKindOfClass:[GenericSphere class]])
		{
			currentGenericSphere = objBeingEdited;
			[self refreshGenericSphere:self];
			[genericSpherePalette makeKeyAndOrderFront:genericSpherePalette];
		}
		else if ([objBeingEdited isKindOfClass:[GenericTorus class]])
		{
			currentGenericTorus = objBeingEdited;
			[self refreshGenericTorus:self];
			[genericTorusPalette makeKeyAndOrderFront:genericTorusPalette];
		}
		else if ([objBeingEdited isKindOfClass:[GenericTaperedCylinder class]])
		{
			currentGenericTaperedCylinder = objBeingEdited;
			[self refreshGenericTaperedCylinder:self];
			[genericTaperedCylinderPalette makeKeyAndOrderFront:genericTaperedCylinderPalette];
		}
		else if ([objBeingEdited isKindOfClass:[Material class]])
		{
			currentMaterial = objBeingEdited;
			[self refreshMaterial:self];
			[materialPalette makeKeyAndOrderFront:materialPalette];
		}
	}
}
- (IBAction)editMaterial:(id)sender
{
	//Tag on the material button that was pressed enables us to decide which type of object to load the material for.
	switch([sender tag])
	{
		case 0: currentMaterial = [currentGenericBox material]; break;
		case 1: currentMaterial = [currentGenericSphere material]; break;
		case 2: currentMaterial = [currentGenericTorus material]; break;
		case 3: currentMaterial = [currentPlane material]; break;
		case 4: currentMaterial = [currentSphere material]; break;
		case 5: currentMaterial = [currentTriangle material]; break;
		case 6: currentMaterial = [currentGenericTaperedCylinder material]; break;
	}
	[self refreshMaterial:self];
	[materialPalette makeKeyAndOrderFront:materialPalette];
}


- (IBAction)refreshSphere:(id)sender
{
	[sphereName setStringValue:[currentSphere name]];
	[[sphereCentreMatrix cellAtRow:0 column:0] setFloatValue:[[currentSphere centre] x]];
	[[sphereCentreMatrix cellAtRow:1 column:0] setFloatValue:[[currentSphere centre] y]];
	[[sphereCentreMatrix cellAtRow:2 column:0] setFloatValue:[[currentSphere centre] z]];
	[sphereRadius setFloatValue:[currentSphere radius]];
	[sphereMaterial setStringValue:[[currentSphere material] name]];
}

- (IBAction)refreshPlane:(id)sender
{
	[planeName setStringValue:[currentPlane name]];
	[[planePointMatrix cellAtRow:0 column:0] setFloatValue:[[currentPlane point] x]];
	[[planePointMatrix cellAtRow:1 column:0] setFloatValue:[[currentPlane point] y]];
	[[planePointMatrix cellAtRow:2 column:0] setFloatValue:[[currentPlane point] z]];
	[[planeNormalMatrix cellAtRow:0 column:0] setFloatValue:[[currentPlane normal] x]];
	[[planeNormalMatrix cellAtRow:1 column:0] setFloatValue:[[currentPlane normal] y]];
	[[planeNormalMatrix cellAtRow:2 column:0] setFloatValue:[[currentPlane normal] z]];
	[planeMaterial setStringValue:[[currentPlane material] name]];
}

- (IBAction)refreshCamera:(id)sender
{
	[cameraName setStringValue:[currentCamera name]];
	[[cameraLocationMatrix cellAtRow:0 column:0] setFloatValue:[[currentCamera location] x]];
	[[cameraLocationMatrix cellAtRow:1 column:0] setFloatValue:[[currentCamera location] y]];
	[[cameraLocationMatrix cellAtRow:2 column:0] setFloatValue:[[currentCamera location] z]];
	[[cameraLookAtMatrix cellAtRow:0 column:0] setFloatValue:[[currentCamera lookAt] x]];
	[[cameraLookAtMatrix cellAtRow:1 column:0] setFloatValue:[[currentCamera lookAt] y]];
	[[cameraLookAtMatrix cellAtRow:2 column:0] setFloatValue:[[currentCamera lookAt] z]];
	[cameraD setFloatValue:[currentCamera d]];
	[cameraTwist setFloatValue:[currentCamera twist]];
}

- (IBAction)refreshLight:(id)sender
{
	[lightName setStringValue:[currentLight name]];
	[[lightLocationMatrix cellAtRow:0 column:0] setFloatValue:[[currentLight location] x]];
	[[lightLocationMatrix cellAtRow:1 column:0] setFloatValue:[[currentLight location] y]];
	[[lightLocationMatrix cellAtRow:2 column:0] setFloatValue:[[currentLight location] z]];
	[lightIntensity setFloatValue:[currentLight intensity]];
	[lightCastsShadow setState:[currentLight castsShadow]];
	[lightColour setColor:[NSColor colorWithCalibratedRed:
		[[currentLight colour] red] green:[[currentLight colour] green] blue:[[currentLight colour] blue] alpha:1.0]];
}

- (IBAction)refreshTriangle:(id)sender
{
	[triangleName setStringValue:[currentTriangle name]];
	[[triangleVertexMatrix cellAtRow:0 column:0] setFloatValue:[[currentTriangle vertexA] x]];
	[[triangleVertexMatrix cellAtRow:1 column:0] setFloatValue:[[currentTriangle vertexA] y]];
	[[triangleVertexMatrix cellAtRow:2 column:0] setFloatValue:[[currentTriangle vertexA] z]];
	[[triangleVertexMatrix cellAtRow:0 column:1] setFloatValue:[[currentTriangle vertexB] x]];
	[[triangleVertexMatrix cellAtRow:1 column:1] setFloatValue:[[currentTriangle vertexB] y]];
	[[triangleVertexMatrix cellAtRow:2 column:1] setFloatValue:[[currentTriangle vertexB] z]];
	[[triangleVertexMatrix cellAtRow:0 column:2] setFloatValue:[[currentTriangle vertexC] x]];
	[[triangleVertexMatrix cellAtRow:1 column:2] setFloatValue:[[currentTriangle vertexC] y]];
	[[triangleVertexMatrix cellAtRow:2 column:2] setFloatValue:[[currentTriangle vertexC] z]];
	[triangleMaterial setStringValue:[[currentTriangle material] name]];
}

- (IBAction)refreshGenericBox:(id)sender
{
	[genericBoxName setStringValue:[currentGenericBox name]];
	[[genericBoxVertexAMatrix cellAtRow:0 column:0] setFloatValue:[[currentGenericBox vertexA] x]];
	[[genericBoxVertexAMatrix cellAtRow:1 column:0] setFloatValue:[[currentGenericBox vertexA] y]];
	[[genericBoxVertexAMatrix cellAtRow:2 column:0] setFloatValue:[[currentGenericBox vertexA] z]];
	[[genericBoxVertexBMatrix cellAtRow:0 column:0] setFloatValue:[[currentGenericBox vertexB] x]];
	[[genericBoxVertexBMatrix cellAtRow:1 column:0] setFloatValue:[[currentGenericBox vertexB] y]];
	[[genericBoxVertexBMatrix cellAtRow:2 column:0] setFloatValue:[[currentGenericBox vertexB] z]];
	[genericBoxMaterial setStringValue:[[currentGenericBox material] name]];
}

- (IBAction)refreshGenericSphere:(id)sender
{
	[genericSphereName setStringValue:[currentGenericSphere name]];
	[genericSphereMaterial setStringValue:[[currentGenericSphere material] name]];
}

- (IBAction)refreshGenericTorus:(id)sender
{
	[genericTorusName setStringValue:[currentGenericTorus name]];
	[genericTorusMajorRadius setFloatValue:[currentGenericTorus major]];
	[genericTorusMinorRadius setFloatValue:[currentGenericTorus minor]];
	[genericTorusMaterial setStringValue:[[currentGenericTorus material] name]];
}

- (IBAction)refreshGenericTaperedCylinder:(id)sender
{
	[genericTaperedCylinderName setStringValue:[currentGenericTaperedCylinder name]];
	[genericTaperedCylinderCapRadius setFloatValue:[currentGenericTaperedCylinder capRadius]];
	[genericTaperedCylinderMaterial setStringValue:[[currentGenericTaperedCylinder material] name]];
}

- (IBAction)refreshMaterial:(id)sender
{
	[materialName setStringValue:[currentMaterial name]];
	[[materialReflectionMatrix cellAtRow:0 column:0] setFloatValue:[currentMaterial globalReflection]];
	[[materialReflectionMatrix cellAtRow:1 column:0] setFloatValue:[currentMaterial diffuseReflection]];
	[[materialReflectionMatrix cellAtRow:2 column:0] setFloatValue:[currentMaterial specularReflection]];
	[[materialReflectionMatrix cellAtRow:3 column:0] setFloatValue:[currentMaterial specularExponent]];
	[[materialReflectionMatrix cellAtRow:4 column:0] setFloatValue:[currentMaterial globalTransmission]];
	[[materialReflectionMatrix cellAtRow:5 column:0] setFloatValue:[currentMaterial refractiveIndex]];
	[materialColour setColor:[NSColor colorWithCalibratedRed:
				[[currentMaterial diffuseColour] red] green:[[currentMaterial diffuseColour] green] blue:[[currentMaterial diffuseColour] blue] alpha:1.0]];
	[materialPalette makeKeyAndOrderFront:materialPalette];
}

- (IBAction)saveMaterial:(id)sender
{
	[currentMaterial setName:[materialName stringValue]];
	[currentMaterial setGlobalReflection:[[materialReflectionMatrix cellAtRow:0 column:0] floatValue]];
	[currentMaterial setDiffuseReflection:[[materialReflectionMatrix cellAtRow:1 column:0] floatValue]];
	[currentMaterial setSpecularReflection:[[materialReflectionMatrix cellAtRow:2 column:0] floatValue]];
	[currentMaterial setSpecularExponent:[[materialReflectionMatrix cellAtRow:3 column:0] floatValue]];
	[currentMaterial setGlobalTransmission:[[materialReflectionMatrix cellAtRow:4 column:0] floatValue]];
	[currentMaterial setRefractiveIndex:[[materialReflectionMatrix cellAtRow:5 column:0] floatValue]];
	[[currentMaterial diffuseColour] setWithValuesNSColor:[materialColour color]];
}

- (IBAction)saveSphere:(id)sender
{
	Vector3D *temp = [[Vector3D alloc] init];

	[currentSphere setObjectName:[sphereName stringValue]];
	[currentSphere setCentre:[temp initWithValuesX:[[sphereCentreMatrix cellAtRow:0 column:0] floatValue]
								y:[[sphereCentreMatrix cellAtRow:1 column:0] floatValue]
								z:[[sphereCentreMatrix cellAtRow:2 column:0] floatValue]
								]];
	[currentSphere setRadius:[sphereRadius floatValue]];
	[currentSphere setMaterialWithName:[sphereMaterial stringValue] materials:[theWorld materials]];
	[temp release];
	[self refreshSphere:self];
}

- (IBAction)savePlane:(id)sender
{		
	[currentPlane setObjectName:[planeName stringValue]];
	[currentPlane setPointWithValuesX:[[planePointMatrix cellAtRow:0 column:0] floatValue]
								y:[[planePointMatrix cellAtRow:1 column:0] floatValue]
								z:[[planePointMatrix cellAtRow:2 column:0] floatValue]
								];
	[currentPlane setNormalWithValuesX:[[planeNormalMatrix cellAtRow:0 column:0] floatValue]
								y:[[planeNormalMatrix cellAtRow:1 column:0] floatValue]
								z:[[planeNormalMatrix cellAtRow:2 column:0] floatValue]
								];
	[currentPlane setMaterialWithName:[planeMaterial stringValue] materials:[theWorld materials]];
	[self refreshPlane:self];
}

- (IBAction)saveCamera:(id)sender
{
	Vector3D *tempVector = [[Vector3D alloc] init];

	[currentCamera setLocation:[tempVector initWithValuesX:[[cameraLocationMatrix cellAtRow:0 column:0] floatValue]
						y:[[cameraLocationMatrix cellAtRow:1 column:0] floatValue]
						z:[[cameraLocationMatrix cellAtRow:2 column:0] floatValue]
						]];
	[currentCamera setLookAt:[tempVector initWithValuesX:[[cameraLookAtMatrix cellAtRow:0 column:0] floatValue]
						y:[[cameraLookAtMatrix cellAtRow:1 column:0] floatValue]
						z:[[cameraLookAtMatrix cellAtRow:2 column:0] floatValue]
						]];
	[currentCamera setD:[cameraD floatValue]];
	[currentCamera setTwist:[cameraTwist floatValue]];
}

- (IBAction)saveLight:(id)sender
{
	Vector3D *tempVector = [[Vector3D alloc] init];
	
	[currentLight setLocationWithVector:[tempVector initWithValuesX:[[lightLocationMatrix cellAtRow:0 column:0] floatValue]
						y:[[lightLocationMatrix cellAtRow:1 column:0] floatValue]
						z:[[lightLocationMatrix cellAtRow:2 column:0] floatValue]
						]];
	[currentLight setName:[lightName stringValue]];
	[currentLight setIntensity:[lightIntensity floatValue]];
	[currentLight setCastsShadow:[lightCastsShadow state]];
	[[currentLight colour] setWithValuesNSColor:[lightColour color]];
}

- (IBAction)saveTriangle:(id)sender
{	
	[currentTriangle setObjectName:[triangleName stringValue]];
	[currentTriangle setVertexA:[Vector3D vector3DWithValuesX:[[triangleVertexMatrix cellAtRow:0 column:0] floatValue]
								y:[[triangleVertexMatrix cellAtRow:1 column:0] floatValue]
								z:[[triangleVertexMatrix cellAtRow:2 column:0] floatValue]
								]];
	[currentTriangle setVertexB:[Vector3D vector3DWithValuesX:[[triangleVertexMatrix cellAtRow:0 column:1] floatValue]
								y:[[triangleVertexMatrix cellAtRow:1 column:1] floatValue]
								z:[[triangleVertexMatrix cellAtRow:2 column:1] floatValue]
								]];
	[currentTriangle setVertexC:[Vector3D vector3DWithValuesX:[[triangleVertexMatrix cellAtRow:0 column:2] floatValue]
								y:[[triangleVertexMatrix cellAtRow:1 column:2] floatValue]
								z:[[triangleVertexMatrix cellAtRow:2 column:2] floatValue]
								]];
	[currentTriangle setMaterialWithName:[triangleMaterial stringValue] materials:[theWorld materials]];
	[self refreshTriangle:self];
}

- (IBAction)saveGenericBox:(id)sender
{	
	[currentGenericBox setObjectName:[genericBoxName stringValue]];
	[currentGenericBox setVertexA:[Vector3D vector3DWithValuesX:[[genericBoxVertexAMatrix cellAtRow:0 column:0] floatValue]
								y:[[genericBoxVertexAMatrix cellAtRow:1 column:0] floatValue]
								z:[[genericBoxVertexAMatrix cellAtRow:2 column:0] floatValue]
								]];
	[currentGenericBox setVertexB:[Vector3D vector3DWithValuesX:[[genericBoxVertexBMatrix cellAtRow:0 column:0] floatValue]
								y:[[genericBoxVertexBMatrix cellAtRow:1 column:0] floatValue]
								z:[[genericBoxVertexBMatrix cellAtRow:2 column:0] floatValue]
								]];
	[currentGenericBox setMaterialWithName:[genericBoxMaterial stringValue] materials:[theWorld materials]];
	[self refreshGenericBox:self];
}

- (IBAction)saveGenericSphere:(id)sender
{	
	[currentGenericSphere setObjectName:[genericSphereName stringValue]];
	[currentGenericSphere setMaterialWithName:[genericSphereMaterial stringValue] materials:[theWorld materials]];
	[self refreshGenericSphere:self];
}

- (IBAction)saveGenericTorus:(id)sender
{	
	[currentGenericTorus setObjectName:[genericTorusName stringValue]];
	[currentGenericTorus setMajor:[genericTorusMajorRadius floatValue]];
	[currentGenericTorus setMinor:[genericTorusMinorRadius floatValue]];
	[currentGenericTorus setMaterialWithName:[genericTorusMaterial stringValue] materials:[theWorld materials]];
	[self refreshGenericTorus:self];
}

- (IBAction)saveGenericTaperedCylinder:(id)sender
{	
	[currentGenericTaperedCylinder setObjectName:[genericTaperedCylinderName stringValue]];
	[currentGenericTaperedCylinder setCapRadius:[genericTaperedCylinderCapRadius floatValue]];
	[currentGenericTaperedCylinder setMaterialWithName:[genericTaperedCylinderMaterial stringValue] materials:[theWorld materials]];
	[self refreshGenericTaperedCylinder:self];
}

- (IBAction)newSphere:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addSphere];
		[object3DView reloadData];
	}
}

- (IBAction)newPlane:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addPlane];
		[object3DView reloadData];
	}
}

- (IBAction)newLight:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addLight];
		[object3DView reloadData];
	}
}

- (IBAction)newTriangle:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addTriangle];
		[object3DView reloadData];
	}

}

- (IBAction)newGenericBox:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addGenericBox];
		[object3DView reloadData];
	}

}

- (IBAction)newGenericSphere:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addGenericSphere];
		[object3DView reloadData];
	}

}

- (IBAction)newGenericTorus:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addGenericTorus];
		[object3DView reloadData];
	}

}

- (IBAction)newGenericTaperedCylinder:(id)sender
{
	if (sceneLoaded)
	{
		[theWorld addGenericTaperedCylinder];
		[object3DView reloadData];
	}

}

@end
