//
//  GenericBox.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Thu Sep 20 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "GenericBox.h"


@implementation GenericBox

// Class methods
+ (GenericBox *)genericBoxWithDictionary:(NSDictionary *)dictionary
{	return [[[GenericBox alloc] initWithDictionary:dictionary] autorelease];	}

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		vertexA = [[Vector3D alloc] init];
		vertexB = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		vertexA = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Vertex A"]];
		vertexB = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Vertex B"]];
		return self;
	}
	return nil; // something went wrong.
}

// copyWithZone is needed for use by NSOutlineView which is used in our editor drawer. All classes which appear in the NSOutlineView will implement this method.
- (id)copyWithZone:(NSZone *)zone	{	return [[GenericBox allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

- (void)dealloc
{
	// dealloc memory created by this class
	[vertexA release];
	[vertexB release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[vertexA asArray] forKey:@"Vertex A"];
	[dictionary setObject:[vertexB asArray] forKey:@"Vertex B"];
	[dictionary setObject:@"Generic Box" forKey:@"Object Type"];
	return dictionary;
}

// Getters
- (Vector3D *)vertexA	{	return vertexA;		}
- (Vector3D *)vertexB	{	return vertexB;		}

// Setters
- (void)setVertexA:(Vector3D *)newVertex	{	[vertexA assign:newVertex];		}
- (void)setVertexB:(Vector3D *)newVertex	{	[vertexB assign:newVertex];		}

// This transforms the ray then passes that to hitBox which does the actual intersection
- (BOOL)intersects:(Ray *)theRay
{
	Ray *transRay = [[Ray alloc] initWithRay:[self transformRay:theRay]];
	BOOL intersects = [self hitBox:transRay];

	if (intersects)
	{
		[hitPoint assign:[[theRay origin] addVector:[[theRay direction] multiplyByFloat:[transRay t]]]];
		[theRay setT:[transRay t]];
	}
	[transRay release];
	return intersects;
}

// This does the actual intersection with the box, after intersect has transformed the ray
- (BOOL)hitBox:(Ray *)ray
{
	float tHit;
	float tIn = -HUGE_VAL;
	float tOut = HUGE_VAL;
	float numerator, denominator;
	int face, faceIn, faceOut;
	float max[3] =
	{
		[vertexA x] >= [vertexB x] ? [vertexA x] : [vertexB x],
		[vertexA y] >= [vertexB y] ? [vertexA y] : [vertexB y],
		[vertexA z] >= [vertexB z] ? [vertexA z] : [vertexB z]
	};
	float min[3] =
	{
		[vertexA x] < [vertexB x] ? [vertexA x] : [vertexB x],
		[vertexA y] < [vertexB y] ? [vertexA y] : [vertexB y],
		[vertexA z] < [vertexB z] ? [vertexA z] : [vertexB z]
	};
	
	for (face = 1; face <= 6; face++)
	{
		switch (face)
		{
			case 1:		// +x face
				numerator = max[0] - [[ray origin] x];
				denominator = [[ray direction] x];
				break;
			case 2:		// +y face
				numerator = max[1] - [[ray origin] y];
				denominator = [[ray direction] y];
				break;
			case 3:		// -x face
				numerator = -min[0] + [[ray origin] x];
				denominator = -[[ray direction] x];
				break;
			case 4:		// -y face
				numerator = -min[1] + [[ray origin] y];
				denominator = -[[ray direction] y];
				break;
			case 5:		// +z face
				numerator = max[2] - [[ray origin] z];
				denominator = [[ray direction] z];
				break;
			case 6:		// -z face
				numerator = -min[2] + [[ray origin] z];
				denominator = -[[ray direction] z];
				break;
		}	// end switch

		// Ray is parallel to a face
		if (fabs (denominator) == 0.0)
		{
			if (numerator < 0.0)	// Ray is out
				return NO;
		}
		else
		{
			tHit = numerator / denominator;
			if  (denominator > 0.0)		// Exiting
			{
				if (tHit < tOut)	// A new closer exit
				{
					tOut = tHit;
					faceOut = face;
				}
			}
			else						// Entering
			{
				if (tHit > tIn)		// A new further entrance
				{
					tIn = tHit;
					faceIn = face;
				}
			}
		}
		
		if  (tIn > tOut)
			return NO;		// Ray misses box
	}	// end for

	if (tIn > EPSILON)
	{
		[ray setT:tIn];
		faceHit = faceIn;
		return YES;
	}
	if (tOut > EPSILON)
	{
		[ray setT:tOut];
		faceHit = faceOut;
		return YES;
	}

	return NO;
}

- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint
{
	Vector3D *normal;
	switch (faceHit)
	{
		case 1:
			normal = [[Vector3D alloc] initWithValuesX:1 y:0 z:0];
			break;
		case 2:
			normal = [[Vector3D alloc] initWithValuesX:0 y:1 z:0];
			break;
		case 3:
			normal = [[Vector3D alloc] initWithValuesX:-1 y:0 z:0];
			break;
		case 4:
			normal = [[Vector3D alloc] initWithValuesX:0 y:-1 z:0];
			break;
		case 5:
			normal = [[Vector3D alloc] initWithValuesX:0 y:0 z:1];
			break;
		case 6:
			normal = [[Vector3D alloc] initWithValuesX:0 y:0 z:-1];
			break;
	}
	[normal autorelease];
	return [normal transformAsNormal:invTransformation];
}

//Debugging
- (void)logValues
{
	[super logValues];
	NSLog(@"Vertex A: (%1.3f, %1.3f, %1.3f), Vertex B: (%1.3f, %1.3f, %1.3f)", [vertexA x], [vertexA y], [vertexA z], [vertexB x], [vertexB y], [vertexB z]);
}

@end
