//
//  GenericObject.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Thu Sep 20 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "GenericObject.h"


@implementation GenericObject
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		transformations = [[NSMutableArray alloc] init];
		invTransformation = [[Transformation alloc] initAsIdentity];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		int i;
		NSArray *tempArray = [[NSArray alloc] initWithArray:[dictionary valueForKey:@"Transformations"]];
		transformations = [[NSMutableArray alloc] init];
		invTransformation = [[Transformation alloc] initAsIdentity];

		for (i = 0; i<[tempArray count]; i++)
		{
			// We need to test what type of transformation it is so we can later create a GUI for it.
			if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Translation"])
			{
				Translation *tempTransformation = [[Translation alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Pitch"])
			{
				Pitch *tempTransformation = [[Pitch alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Yaw"])
			{
				Yaw *tempTransformation = [[Yaw alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Roll"])
			{
				Roll *tempTransformation = [[Roll alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Scale"])
			{
				Scale *tempTransformation = [[Scale alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Shear"])
			{
				Shear *tempTransformation = [[Shear alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
			else if ([[[tempArray objectAtIndex:i] valueForKey:@"Transformation Type"] isEqualToString:@"Reflection"])
			{
				Reflection *tempTransformation = [[Reflection alloc] initWithDictionary:[tempArray objectAtIndex:i]];
				[transformations addObject:tempTransformation];
				[tempTransformation release];
			}
		}
		[tempArray release];
		[self calculateInvTransformation];
		return self;
	}
	return nil; // something went wrong.
}

- (void)dealloc
{
	// dealloc memory created by this class
	[transformations release];
	[invTransformation release];
	//***** invTextureTransformation belongs to Object3D... we got confused before
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary
{
	int i;
	NSMutableDictionary *dictionary = [super asDictionary];
	NSMutableArray *transDictArray = [[NSMutableArray alloc] init];
	for (i = 0; i < [transformations count]; i++)
	{
		[transDictArray addObject:[[transformations objectAtIndex:i] asDictionary]];
	}
	[dictionary setObject:transDictArray forKey:@"Transformations"];
	[transDictArray release];
	return dictionary;
}

// Getters
- (NSMutableArray *)transformations						{	return transformations;		}
- (Transformation *)transformationAtIndex:(int)index	{	return [transformations objectAtIndex:index];	}
- (Transformation *)invTransformation					{	return invTransformation;	}

// Setters
// This sets the whole transformation matrix, basically just here for completeness, not sure we will ever use this
- (void)setTransformations:(NSArray *)newTransformations	{	[transformations setArray:newTransformations];	}

// Transformation array manipulation routines
// This is used when you want to add a transformation to the transformation array
- (void)addTransformation:(Transformation *)newTransformation atIndex:(unsigned)index	{	[transformations insertObject:newTransformation atIndex:index];		}
- (void)deleteTransformationAtIndex:(unsigned)index										{	[transformations removeObjectAtIndex:index];						}

// This calculates both the full inverse transformation and the possibly lesser transformation needed for textures
- (void)calculateInvTransformation
{
	int i;

	// If this is not the first time we have called this function we want to clear invTransformation before performing calculation on it
	if (invTransformation != nil)
	{
		[invTransformation release];
		invTransformation = nil;
	}
	invTransformation = [[Transformation alloc] initAsIdentity];
	if (invTextureTransformation != nil)
	{
		[invTextureTransformation release];
		invTextureTransformation = nil;
	}
	invTextureTransformation = [[Transformation alloc] initAsIdentity];
//	[invTransformation logValues];
//	NSLog (@"Number of transformations in array: %d", [transformations count]);
	for (i = [transformations count] - 1; i >= 0; i--)
	{
		[invTransformation assign:[invTransformation multiplyByTransformation:[transformations objectAtIndex:i]]];
		// Some textures might not want to follow the object, so we don't add them to the inverse texture matrix. We do not support intrinsic texture transformations
		if ([[transformations objectAtIndex:i] propagatesToTextures])
			[invTextureTransformation assign:[invTextureTransformation multiplyByTransformation:[transformations objectAtIndex:i]]];
		[[transformations objectAtIndex:i] logValues];
//		[invTransformation logValues];
	}
}

// This is called when we want to intersect a ray with an object
- (Ray *)transformRay:(Ray *)ray
{
	Ray *tempRay = [[[Ray alloc] init] autorelease];
	[tempRay setOriginWithVector:[[ray origin] transformAsPoint:invTransformation]];
	[tempRay setDirectionWithVector:[[ray direction] transformAsVector:invTransformation]];
	return tempRay;
}

// returns a pointer to the colour
- (RGBVector *)diffuseColour:(Vector3D *)intersect	{	return [material diffuseColour:[intersect transformAsPoint:invTextureTransformation]];		}
- (float)diffuseReflection:(Vector3D *)intersect	{	return [material diffuseReflection:[intersect transformAsPoint:invTextureTransformation]];	}
- (float)specularReflection:(Vector3D *)intersect	{	return [material specularReflection:[intersect transformAsPoint:invTextureTransformation]];	}
- (BOOL)isDiffuseReflective:(Vector3D *)intersect	{	return [material isDiffuseReflective:[intersect transformAsPoint:invTextureTransformation]];	}
- (BOOL)isSpecularReflective:(Vector3D *)intersect	{	return [material isSpecularReflective:[intersect transformAsPoint:invTextureTransformation]];	}
- (BOOL)isTransparent:(Vector3D *)intersect			{	return [material isTransparent:[intersect transformAsPoint:invTextureTransformation]];		}
- (BOOL)isReflective:(Vector3D *)intersect			{	return [material isReflective:[intersect transformAsPoint:invTextureTransformation]];		}

//Debugging
- (void)logValues
{
	int i;
	[super logValues];
	for (i=0; i<[transformations count]; i++)
	{
		[[transformations objectAtIndex:i] logValues];
	}
}

@end
