//
//  GenericSphere.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Thu Sep 20 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "GenericSphere.h"


@implementation GenericSphere

// Class methods
+ (GenericSphere *)genericSphereWithDictionary:(NSDictionary *)dictionary
{	return [[[GenericSphere alloc] initWithDictionary:dictionary] autorelease];		}

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		normal = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		normal = [[Vector3D alloc] init];
//		[self logValues];
		return self;
	}
	return nil; // something went wrong.
}

// copyWithZone is needed for use by NSOutlineView which is used in our editor drawer. All classes which appear in the NSOutlineView will implement this method.
- (id)copyWithZone:(NSZone *)zone	{	return [[GenericSphere allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

- (void)dealloc
{
	// dealloc memory created by this class
	[normal release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:@"Generic Sphere" forKey:@"Object Type"];
	return dictionary;
}

// implementation of the abstract method in Object3D. This returns true if theRay intersects with the Sphere
- (BOOL)intersects:(Ray *)theRay
{
	// declare local variables and init them to the discriminant equation
	float t, a, b, c, discrim, e, denom;
	Ray *transRay = [[self transformRay:theRay] retain];
	a = [[transRay direction] dotProduct:[transRay direction]];
	b = [[[transRay origin] multiplyByFloat:2.0] dotProduct:[transRay direction]];
	c = [[transRay origin] dotProduct:[transRay origin]] - 1;
	discrim = b * b - 4.0 * a * c;

	// if discrim is < 0 then the ray did not intersect
	if (discrim >= 0)
	{
		e = sqrt(discrim);
		denom = 2.0 * a;
		t = (-b - e) / denom;	// Smaller root
		
		if (t > EPSILON)
		{
			// set the distance of the ray
			[theRay setT:t];
			[transRay setT:t];
			// The ray did intersect and we are storing the intersect point
			[hitPoint assign:[theRay hitPoint]];
			[normal assign:[[transRay hitPoint] transformAsNormal:invTransformation]];
			[normal normalise];
			[transRay release];
			return YES;
		}
		
		t = (-b + e) / denom;	// Larger root
		
		if (t > EPSILON)
		{
			// set the distance of the ray
			[theRay setT:t];
			[transRay setT:t];
			// The ray did intersect and we are storing the intersect point
			[hitPoint assign:[theRay hitPoint]];
			[normal assign:[[transRay hitPoint] transformAsNormal:invTransformation]];
			[normal normalise];
			[transRay release];
			return YES;
		}
	}
	
	[transRay release];
	return NO;
}

// implementation of the abstract method in Object3D
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint	{	return normal;	}

@end
