//
//  GenericTaperedCylinder.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Tue Oct 24 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "GenericTaperedCylinder.h"

@implementation GenericTaperedCylinder

// Class methods
+ (GenericTaperedCylinder *)genericTaperedCylinderWithDictionary:(NSDictionary *)dictionary
{	return [[[GenericTaperedCylinder alloc] initWithDictionary:dictionary] autorelease];		}

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		normal = [[Vector3D alloc] init];
		// A generic cylinder
		capRadius = 1;
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		capRadius = [[dictionary valueForKey:@"Cap Radius"] floatValue];
		normal = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// copyWithZone is needed for use by NSOutlineView which is used in our editor drawer. All classes which appear in the NSOutlineView will implement this method.
- (id)copyWithZone:(NSZone *)zone	{	return [[GenericTaperedCylinder allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

- (void)dealloc
{
	// dealloc memory created by this class
	[normal release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[NSNumber numberWithFloat:capRadius] forKey:@"Cap Radius"];
	[dictionary setObject:@"Generic Tapered Cylinder" forKey:@"Object Type"];
	return dictionary;
}

// Getters
- (float)capRadius	{	return capRadius;		}

// Setters
- (void)setCapRadius:(float)newCapRadius	{	capRadius = newCapRadius;	}

// implementation of the abstract method in Object3D. This returns true if theRay intersects with the Sphere
- (BOOL)intersects:(Ray *)theRay
{
	float A, B, C;
	float dx, dy, dz, ox, oy, oz;
	float t, minRoot = HUGE_VAL;
	float fDir, fStart, discrim;

	float sm = capRadius - 1;
	Ray *transRay = [[Ray alloc] initWithRay:[self transformRay:theRay]];

	dx = [[transRay direction] x];
	dy = [[transRay direction] y];
	dz = [[transRay direction] z];
	ox = [[transRay origin] x];
	oy = [[transRay origin] y];
	oz = [[transRay origin] z];

	// I think this is the last time we need it
	[transRay release];

	fDir = sm * dz;
	fStart = sm * oz + 1;

	A = SQR(dx) + SQR(dy) - SQR(fDir);
	B = ox * dx + oy * dy - fDir * fStart;
	C = SQR(ox) + SQR(oy) - SQR(fStart);

	discrim = B * B - A * C;

	// Test if the ray hits the cylinder
	if (discrim > 0.0)	// Can take square root
	{
		float zHit;
		float disc_root = (float)sqrt(discrim);

		t = (-B - disc_root) / A;	// Earlier hit
		zHit = oz + dz * t;
		if (t > EPSILON && t < minRoot && zHit <= 1.0 && zHit >= 0.0)
		{
//			NSLog(@"Earlier wall hit t: %1.4f, zHit: %1.4f, minRoot: %1.4f", t, zHit, minRoot);
			minRoot = t;
			surface = WALL;
		}

		t = (-B + disc_root) / A;	// Second hit
		zHit = oz + dz * t;
		if (t > EPSILON && t < minRoot && zHit <= 1.0 && zHit >= 0.0)
		{
//			NSLog(@"Later wall hit t: %1.4f, zHit: %1.4f, minRoot: %1.4f", t, zHit, minRoot);
			minRoot = t;
			surface = WALL;
		}
	}		// End if (discrim > 0.0)

	// Test hitting the base z = 0
	t = -oz / dz;	// Hit time at z = 0 plane
	if (t > EPSILON && t < minRoot && (SQR(ox + dx * t) + SQR(oy + dy * t)) < 1.0)		// Within disc of base
	{
//		NSLog(@"Base hit t: %1.4f, minRoot: %1.4f", t, minRoot);
		minRoot = t;
		surface = BASE;
	}

	// Test hitting the cap at z = 1
	t = (1 - oz) / dz;	// Hit time at z = 1 plane
	if (t > EPSILON && t < minRoot && (SQR(ox + dx * t) + SQR(oy + dy * t)) < SQR(capRadius))	// Within disc of cap
	{
//		NSLog(@"Cap hit t: %1.4f, minRoot: %1.4f", t, minRoot);
		minRoot = t;
		surface = CAP;
	}

	// Did the ray actually hit anything
	if (minRoot == HUGE_VAL)
		return NO;

	// set the distance of the ray
	[theRay setT:minRoot];
	[hitPoint assign:[theRay hitPoint]];
//	[hitPoint assign:[[theRay origin] addVector:[[theRay direction] multiplyByFloat:minRoot]]];
	
	// Since we did hit something, we need to set the normal
	if (surface == WALL)
		[normal setWithValuesX:(ox + dx * minRoot) y:(oy + dy * minRoot) z:(-sm * (1 + sm * (oz + dz * minRoot)))];
	else if (surface == BASE)
		[normal setWithValuesX:0 y:0 z:-1];
	else
		[normal setWithValuesX:0 y:0 z:1];

//	NSLog(@"We hit surface %d, minRoot: %1.4f, with normal:", surface, minRoot);
//	[normal logValues];

	[normal assign:[[normal transformAsNormal:invTransformation] unit]];

	return YES;
}

// implementation of the abstract method in Object3D
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint
{	return normal;	}

@end