//
//  Light.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Tue Aug 14 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Light.h"


@implementation Light
// Class methods
+ (Light *)lightWithDictionary:(NSDictionary *)dictionary
{	return [[[Light alloc] initWithDictionary:dictionary] autorelease];		}

+ (Light *)lightWithLinearFrameStart:(Light *)start end:(Light *)end currentFrame:(int)currentFrame totalFrames:(int)totalFrames
{
	return [[[Light alloc] initWithVectorsLocation:
				[Vector3D vector3DWithLinearFrameStart:[start location] end:[end location] currentFrame:currentFrame totalFrames:totalFrames]
				colour:[RGBVector rgbVectorWithLinearFrameStart:[start colour] end:[end colour] currentFrame:currentFrame totalFrames:totalFrames]
				intensity:([start intensity] + (([end intensity] - [start intensity])/(totalFrames - 1))*currentFrame)
				castsShadow:[start castsShadow]
				name:[start name]
			] autorelease];
}

// Yet another boring init
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] init];
		colour = [[RGBVector alloc] init];
		name = [[NSMutableString alloc] init];
		intensity = 1.0;
		castsShadow = YES;
		return self;
	}
	return nil; // something went wrong.
}

// Initialisation
// Is anyone crazy enough to actually use this ridiculous init routine? Why do you think we are now using dictionaries? Now that we know how to use them, of course
- (id)initWithValuesX:(float)newX y:(float)newY z:(float)newZ red:(float)newRed green:(float)newGreen blue:(float)newBlue intensity:(float)newIntensity
			castsShadow:(BOOL)newCastsShadow name:(NSString *)newName
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] initWithValuesX:newX y:newY z:newZ];
		colour = [[RGBVector alloc] initWithValuesRed:newRed green:newGreen blue:newBlue];
		intensity = newIntensity;
		castsShadow = newCastsShadow;
		name = [[NSString alloc] initWithString:newName];
		return self;
	}
	return nil; // something went wrong.
}


//WARNING>>> THIS INIT USES RETAIN AND THUS SHOULD NOT BE INITED WITH VECTORS THAT ARE REQUIRED TO HAVE DIFFERENT VALUES TO THAT OF THIS LIGHT (for now?...)
- (id)initWithVectorsLocation:(Vector3D *)newLocation colour:(RGBVector *)newColour intensity:(float)newIntensity castsShadow:(BOOL)newCastsShadow
			name:(NSString *)newName
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] initWithVector:newLocation];
		colour = [[RGBVector alloc] initWithRGBVector:newColour];
		intensity = newIntensity;
		castsShadow = newCastsShadow;
		name = [[NSString alloc] initWithString:newName];
		return self;
	}
	return nil; // something went wrong.

}

// Initialisation with dictionary
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		location = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Location"]];
		colour = [[RGBVector alloc] initWithArray:[dictionary valueForKey:@"Colour"]];
		intensity = [[dictionary valueForKey:@"Intensity"] floatValue];
		castsShadow = [[dictionary valueForKey:@"Casts Shadow"] boolValue];
		name = [[NSMutableString alloc] initWithString:[dictionary valueForKey:@"Name"]];
		return self;
	}
	return nil; // something went wrong.
}

// This is the copy constructor. It is needed by NSOutlineView. It copies the object within the same memory space (zone)
- (id)copyWithZone:(NSZone *)zone
{
	return [[Light allocWithZone:zone] initWithValuesX:[location x] y:[location y] z:[location z]
										red:[colour red] green:[colour green] blue:[colour blue] intensity:intensity castsShadow:castsShadow name:name];
}

// Destructor - mighty destructor
- (void)dealloc
{
	[location release];
	[colour release];
	[name release];
	[super dealloc];  //use the super class's method for instance destruction
}

// To save in a scene file
- (NSDictionary *)asDictionary
{
	return [NSDictionary dictionaryWithObjects:
				[NSArray arrayWithObjects:
					[location asArray],
					[colour asArray],
					// Dictionaries will take just a float here but to save it to an XML property list, Apple require it be an NSNumber
					[NSNumber numberWithFloat:intensity],
					[NSNumber numberWithBool:castsShadow],
					name,
					@"Point",
					nil
				]
			forKeys:[NSArray arrayWithObjects:@"Location", @"Colour", @"Intensity", @"Casts Shadow", @"Name", @"Light Type", nil]
			];
}

// Getters
- (Vector3D *)location	{	return location;	}
- (RGBVector *)colour	{	return colour;		}
- (NSString *)name		{	return name;		}
- (float)intensity		{	return intensity;	}
- (BOOL)castsShadow		{	return castsShadow;	}

// Setters
- (void)setLocationWithVector:(Vector3D *)newLocation					{	[location assign:newLocation];					}
- (void)setLocationWithValuesX:(float)newX y:(float)newY z:(float)newZ	{	[location setWithValuesX:newX y:newY z:newZ];	}
- (void)setColourWithVector:(RGBVector *)newColour						{	[colour assign:newColour];						}
- (void)setColourWithValuesRed:(float)newRed green:(float)newGreen blue:(float)newBlue	{	[colour setWithValuesRed:newRed green:newGreen blue:newBlue];	}
- (void)setCastsShadow:(BOOL)rhs										{	castsShadow = rhs;								}
- (void)setName:(NSString *)newName										{	[name setString:newName];						}
- (void)setIntensity:(float)newIntensity								{	intensity = newIntensity;						}

- (RGBVector *)colourWithIntensity										{	return [colour multiplyByFloat:(intensity)];	}

// Debugging
- (void)logValues
{
	NSLog(@"Light %s logged. castsShadow: %b, intensity: %f, location: (%1.3f, %1.3f, %1.3f), colour: (%1.3f, %1.3f, %1.3f)",
			name, castsShadow, intensity, [location x], [location y], [location z], [colour red], [colour green], [colour blue]);
}
@end
