//
//  Material.h
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Sat Aug 25 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

// The base phong material

#import <Foundation/Foundation.h>
#import "RGBVector.h"
#import "Vector3D.h"

@interface Material : NSObject {
	// The various reflective indicies
	float diffuseReflection, specularReflection, globalReflection;
	// The intrinsic colour of the material
	RGBVector *diffuseColour;
	// The name of the material, not currently used
	NSMutableString *name;
	// Determines the size of the specular reflection
	int specularExponent;
	// The relative density of the object compared to a vacuum
	float refractiveIndex;
	// The amount of light which gets transmitted through an object
	float globalTransmission;
}

// Class methods
+ (Material *)materialWithDictionary:(NSDictionary *)dictionary;

// Used when we are loading a scene file
- (id)initWithDictionary:(NSDictionary *)dictionary;
// Used when we are saving to a scene file
- (NSMutableDictionary *)asDictionary;
// Assignment method
- (void)assign:(Material *)rhs;											

// Diffuse reflection at a point, used for textures
- (float)diffuseReflection:(Vector3D *)hitPoint;
// Specular reflection at a point, used for textures
- (float)diffuseReflection;
// Specular reflection, does this material have specular highlights
- (float)specularReflection:(Vector3D *)hitPoint;
// Diffuse reflection, how much of this object is reflected
- (float)specularReflection;
// Global reflection, how much of the world can you see reflected in this object
- (float)globalReflection:(Vector3D *)hitPoint;
- (float)globalReflection;
// Size of specular reflection
- (int)specularExponent:(Vector3D *)hitPoint;
- (int)specularExponent;
// The relative density of the object compared to a vacuum
- (float)refractiveIndex;
// The amount of light which gets transmitted through an object
- (float)globalTransmission:(Vector3D *)hitPoint;
- (float)globalTransmission;
// If the object is diffuse reflective at a point
- (BOOL)isDiffuseReflective:(Vector3D *)hitPoint;
// If the object is specular reflective at a point
- (BOOL)isSpecularReflective:(Vector3D *)hitPoint;
// Combination of the two above
- (BOOL)isReflective:(Vector3D *)hitPoint;
// It the object transparent at this point
- (BOOL)isTransparent:(Vector3D *)hitPoint;
// The colour of the object. Should use a hitpoint for textures. Will get around to that soon...
- (RGBVector *)diffuseColour;
// The colour of the object.
- (RGBVector *)diffuseColour:(Vector3D *)hitPoint;
// The name of the material
- (NSString *)name;

// Setters for the respective variables
- (void)setGlobalReflection:(float)newGlobalReflection;
- (void)setDiffuseReflection:(float)newDiffuseReflection;
- (void)setSpecularReflection:(float)newSpecularReflection;
- (void)setSpecularExponent:(int)newSpecularExponent;
- (void)setRefractiveIndex:(float)newRefractiveIndex;
- (void)setGlobalTransmission:(float)newGlobalTransmission;
- (void)setDiffuseColour:(RGBVector *)newDiffuseColour;
- (void)setName:(NSString *)newString;
@end
