//
//  Material.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Sat Aug 25 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Material.h"


@implementation Material

// Class methods
+ (Material *)materialWithDictionary:(NSDictionary *)dictionary
{	return [[[Material alloc] initWithDictionary:dictionary] autorelease];		}

// Normal constructor, probably never called, but what the hey
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		diffuseColour = [[RGBVector alloc] init];
		name = [[NSMutableString alloc] init];
		diffuseReflection = 1.0;
		specularReflection = 0.0;
		specularExponent = 0;
		globalReflection = 0.5;
		refractiveIndex = 0.0;
		globalTransmission = 0.0;
		return self;
	}
	return nil; // something went wrong.
}

// Used when we are loading a scene file
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// initiallise diffuseColour from an array of NSNumbers
		diffuseColour = [[RGBVector alloc] initWithValuesRed:[[[dictionary valueForKey:@"Diffuse Colour"] objectAtIndex:0] floatValue]
									green:[[[dictionary valueForKey:@"Diffuse Colour"] objectAtIndex:1] floatValue]
									blue:[[[dictionary valueForKey:@"Diffuse Colour"] objectAtIndex:2] floatValue]
									];
		// Set the name from the dictionary
		name = [[NSMutableString alloc] initWithString:[dictionary valueForKey:@"Name"]];
		// Set the reflective coefficients from the NSNumbers in the dictionary
		diffuseReflection = [[dictionary valueForKey:@"Diffuse Reflection"] floatValue];
		specularReflection = [[dictionary valueForKey:@"Specular Reflection"] floatValue];
		specularExponent = [[dictionary valueForKey:@"Specular Exponent"] intValue];
		globalReflection = [[dictionary valueForKey:@"Global Reflection"] floatValue];
		refractiveIndex = [[dictionary valueForKey:@"Refractive Index"] floatValue];
		globalTransmission = [[dictionary valueForKey:@"Global Transmission"] floatValue];
		return self;
	}
	return nil; // something went wrong.
}

- (id)copyWithZone:(NSZone *)zone	{	return [[Material allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

// Destructor
- (void)dealloc
{
	[diffuseColour release];
	[name release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// Used when we are saving to a scene file
- (NSMutableDictionary *)asDictionary
{
	return [NSMutableDictionary dictionaryWithObjects:
				[NSArray arrayWithObjects:
					// Create an array with our RGB diffuseColour as NSNumbers
					[NSArray arrayWithObjects:[NSNumber numberWithFloat:[diffuseColour red]],
						[NSNumber numberWithFloat:[diffuseColour green]],
						[NSNumber numberWithFloat:[diffuseColour blue]],
						nil
					],
					// Various reflective coefficients
					[NSNumber numberWithFloat:diffuseReflection],
					[NSNumber numberWithFloat:specularReflection],
					[NSNumber numberWithFloat:globalReflection],
					[NSNumber numberWithInt:specularExponent],
					[NSNumber numberWithFloat:refractiveIndex],
					[NSNumber numberWithFloat:globalTransmission],
					// The name of the material
					name,
					// The type of material
					@"Phong",
					nil
				]
				// Set the keys of the dictionary, they correspond to the lines immediately above
				forKeys:[NSArray arrayWithObjects:@"Diffuse Colour", @"Diffuse Reflection", @"Specular Reflection", @"Global Reflection", @"Specular Exponent", @"Refractive Index", @"Global Transmission", @"Name", @"Material Type", nil]
			];
}

// Assignment method
- (void)assign:(Material *)rhs
{
	[diffuseColour assign:[rhs diffuseColour]];
	diffuseReflection = [rhs diffuseReflection];
	specularReflection = [rhs specularReflection];
	specularExponent = [rhs specularExponent];
	globalReflection = [rhs globalReflection];
	refractiveIndex = [rhs refractiveIndex];
	globalTransmission = [rhs globalTransmission];
}

// Diffuse reflection at a point, used for textures
- (float)diffuseReflection:(Vector3D *)hitPoint		{	return diffuseReflection;	}
// Diffuse reflection, how much of this object is reflected
- (float)diffuseReflection							{	return diffuseReflection;	}
// Specular reflection at a point, used for textures
- (float)specularReflection:(Vector3D *)hitPoint	{	return specularReflection;	}
// Specular reflection, does this material have specular highlights
- (float)specularReflection							{	return specularReflection;	}
// Size of specular reflection
- (int)specularExponent:(Vector3D *)hitPoint		{	return specularExponent;	}
- (int)specularExponent								{	return specularExponent;	}
// Global reflection, how much of the world can you see reflected in this object
- (float)globalReflection:(Vector3D *)hitPoint		{	return globalReflection;	}
- (float)globalReflection							{	return globalReflection;	}
// The relative density of the object compared to a vacuum
- (float)refractiveIndex							{	return refractiveIndex;		}
// The amount of light which gets transmitted through an object
- (float)globalTransmission:(Vector3D *)hitPoint	{	return globalTransmission;	}
- (float)globalTransmission							{	return globalTransmission;	}
// If the object is diffuse reflective at a point, ie diffuseReflection is greater than zero
- (BOOL)isDiffuseReflective:(Vector3D *)hitPoint	{	return (diffuseReflection > 0);		}
// If the object is specular reflective at a point, ie specularReflection is greater than zero
- (BOOL)isSpecularReflective:(Vector3D *)hitPoint	{	return (specularReflection > 0);	}
// Combination of the two above
- (BOOL)isReflective:(Vector3D *)hitPoint			{	return (diffuseReflection > 0 || specularReflection > 0);	}
// It the object transparent at this point
- (BOOL)isTransparent:(Vector3D *)hitPoint			{	return (globalTransmission > 0);	}
// The colour of the object. Should use a hitpoint for textures. Will get around to that soon...
- (RGBVector *)diffuseColour						{	return diffuseColour;	}
// The colour of the object.
- (RGBVector *)diffuseColour:(Vector3D *)hitPoint	{	return diffuseColour;	}
// The name of the object, not currently used
- (NSString *)name									{	return name;			}

// Setters for the respective variables
- (void)setGlobalReflection:(float)newGlobalReflection		{	globalReflection = newGlobalReflection;		}
- (void)setDiffuseReflection:(float)newDiffuseReflection	{	diffuseReflection = newDiffuseReflection;	}
- (void)setSpecularReflection:(float)newSpecularReflection	{	specularReflection = newSpecularReflection;	}
- (void)setSpecularExponent:(int)newSpecularExponent		{	specularExponent = newSpecularExponent;		}
- (void)setRefractiveIndex:(float)newRefractiveIndex		{	refractiveIndex = newRefractiveIndex;		}
- (void)setGlobalTransmission:(float)newGlobalTransmission	{	globalTransmission = newGlobalTransmission;	}
- (void)setDiffuseColour:(RGBVector *)newDiffuseColour		{	[diffuseColour assign:newDiffuseColour];	}
- (void)setName:(NSString *)newString						{	[name setString:newString];					}
@end
