//
//  NoiseSource.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Oct 19 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "NoiseSource.h"


@implementation NoiseSource

// Class methods
+ (NoiseSource *)noiseSourceWithDictionary:(NSDictionary *)dictionary
{	return [[[NoiseSource alloc] initWithDictionary:dictionary] autorelease];		}

// Normal constructor, probably never called, but what the hey
- (id)init
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		double seed = (double)[[NSDate date] timeIntervalSince1970];
		kTableMask = kTableSize - 1;
		vectorTable = [NSMutableArray arrayWithCapacity:kTableSize];
		[self valueTableInit:seed];
		[self vectorTableInit:seed];

		return self;
	}
	return nil; // something went wrong.
}

// Used when we are loading a scene file
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using our init method
	if (self = [self init])
	{	// self init worked
		numOctaves = [[dictionary valueForKey:@"Octaves"] intValue];
		lacunarity = [[dictionary valueForKey:@"Lacunarity"] floatValue];
		gain = [[dictionary valueForKey:@"Gain"] floatValue];

		return self;
	}
	return nil; // something went wrong.
}

- (id)copyWithZone:(NSZone *)zone	{	return [[NoiseSource allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

// Release all memory
- (void)dealloc
{
	[vectorTable release];
	[super dealloc];  //use the super class's method for instance destruction
}

// Used when we are saving to a scene file
- (NSMutableDictionary *)asDictionary
{
	return [NSMutableDictionary dictionaryWithObjects:
				[NSArray arrayWithObjects:
					[NSNumber numberWithFloat:numOctaves],
					[NSNumber numberWithFloat:lacunarity],
					[NSNumber numberWithFloat:gain],
					// tell the dictionary what type of object this is
					@"Noise Source",
					nil
				]
				// give the above objects keys in the dictionary
				forKeys:[NSArray arrayWithObjects:@"Octaves", @"Lacunarity", @"Gain", @"Noise Type", nil]
			];
}

// Getters
- (int)numOctaves		{	return numOctaves;	}
- (float)lacunarity		{	return lacunarity;	}
- (float)gain			{	return gain;		}

- (unsigned char)perm:(int)x
{
	unsigned char perm[kTableSize] =
			{
				225,155,210,108,175,199,221,144,203,116, 70,213, 69,158, 33,252,
		        5, 82,173,133,222,139,174, 27,  9, 71, 90,246, 75,130, 91,191,
		        169,138,  2,151,194,235, 81,  7, 25,113,228,159,205,253,134,142,
		        248, 65,224,217, 22,121,229, 63, 89,103, 96,104,156, 17,201,129,
		        36,  8,165,110,237,117,231, 56,132,211,152, 20,181,111,239,218,
		        170,163, 51,172,157, 47, 80,212,176,250, 87, 49, 99,242,136,189,
		        162,115, 44, 43,124, 94,150, 16,141,247, 32, 10,198,223,255, 72,
		        53,131, 84, 57,220,197, 58, 50,208, 11,241, 28,  3,192, 62,202,
		        18,215,153, 24, 76, 41, 15,179, 39, 46, 55,  6,128,167, 23,188,
		        106, 34,187,140,164, 73,112,182,244,195,227, 13, 35, 77,196,185,
		        26,200,226,119, 31,123,168,125,249, 68,183,230,177,135,160,180,
		        12,  1,243,148,102,166, 38,238,251, 37,240,126, 64, 74,161, 40,
		        184,149,171,178,101, 66, 29, 59,146, 61,254,107, 42, 86,154,  4,
		        236,232,120, 21,233,209, 45, 98,193,114, 78, 19,206, 14,118,127,
		        48, 79,147, 85, 30,207,219, 54, 88,234,190,122, 95, 67,143,109,
		        137,214,145, 93, 92,100,245,  0,216,186, 60, 83,105, 97,204, 52
			};

//	NSLog (@"perm[%d] = %u", x, perm[x]);
	return perm[x];
}

// Setters
- (void)setNumOctaves:(int)octaves				{	numOctaves = octaves;			}
- (void)setLacunarity:(float)newLacunarity		{	lacunarity = newLacunarity;		}
- (void)setGain:(float)newGain					{	gain = newGain;					}


- (void)valueTableInit:(int)seed
{
	int i;
	srand(seed);
	NSLog (@"ValueTableInit called with seed: %d", seed);
	for (i = 0; i < kTableSize; i++)
		valueTable[i] = 1.0 - 2.0 * rand() / (float)RAND_MAX;

}

- (void)vectorTableInit:(int)seed
{
	int i;
	srand(seed);
	NSLog (@"VectorTableInit called with seed: %d", seed);
	[vectorTable removeAllObjects];
	for (i = 0; i < kTableSize; i++)
	{
		// Find a random vector whose end lies inside the unit sphere
		Vector3D *v = [[Vector3D alloc] init];

		do
		{
			[v setX:1.0 - 2.0 * rand() / (float)RAND_MAX];
			[v setY:1.0 - 2.0 * rand() / (float)RAND_MAX];
			[v setZ:1.0 - 2.0 * rand() / (float)RAND_MAX];
		} while (([v x] * [v x] + [v y] * [v y] + [v z] * [v z]) > 1.0);

		[v normalise];
//		[vectorTable replaceObjectAtIndex:i withObject:v];
		[vectorTable addObject:v];
	}
}

- (float)valueLatticeX:(int)ix y:(int)iy z:(int)iz			{	return valueTable[INDEX(ix,iy,iz)];		}
- (Vector3D *)vectorLatticeX:(int)ix y:(int)iy z:(int)iz	{	return [vectorTable objectAtIndex: INDEX(ix,iy,iz)];	}

- (float)scalarNoise:(Vector3D *)point			{	return 0.0;		}
- (Vector3D *)vectorNoise:(Vector3D *)point		{	return [Vector3D vector3DWithValuesX:0.0 y:0.0 z:0.0];		}

- (float)scalarTurbulence:(Vector3D *)point
{
	int j;
	float turbulence = 0.0, scale = 1.0;

	for (j = 0; j < numOctaves; j++)
	{
		turbulence += fabs([self scalarNoise:[point multiplyByFloat:scale]] / scale);
		scale *= 2;
	}
	
	return (turbulence);
}

- (Vector3D *)vectorTurbulence:(Vector3D *)point	{	return nil;		}

- (float)fractalSum:(Vector3D *)point
{
	float fractalSum = 0.0, scale = 1.0;
	int j;
	
	for (j = 0; j < numOctaves; j++)
	{
		fractalSum += [self scalarNoise:[point multiplyByFloat:scale]] / scale;
		scale *= 2;
	}
	return fractalSum;
}

- (Vector3D *)vFractalSum:(Vector3D *)point		{	return nil;		}

- (float)fBm:(Vector3D *)point
{
	float sum = 0.0;
	float frequency = 1.0, amplitude = 1.0;
	int j;
	
	for (j = 0; j < numOctaves; j++)
	{
		sum			+= amplitude * [self scalarNoise:[point multiplyByFloat:frequency]];
		frequency	*= lacunarity;
		amplitude	*= gain;
	}
	
	return sum;
}

- (Vector3D *)vfBm:(Vector3D *)point	{	return nil;		}

- (void)logValues
{	NSLog(@"Octaves: %d, Lacunarity: %1.4f, Gain: %1.4f", numOctaves, lacunarity, gain);	}

@end
