//
//  Object.h
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "Ray.h"
#import "Vector3D.h"
#import "RGBVector.h"
#import "Materials.h"
#import "AllTransformations.h"

#define EPSILON 0.001	// To avoid salt and pepper effect, use t > EPSILON

@interface Object3D : NSObject {
	// store the intersection point. This is used only when the scene is rendered and is only relevant for one ray at a time
	Vector3D *hitPoint;
	// stores the Material object associated with this object
	Material *material;
	// stores the object name
	NSMutableString *name;
	// This is so we can calculate the texture transformation in material, obviously it does not get applied to non-generic objects
	Transformation *invTextureTransformation;
	// do we store pitch yaw and roll?
	float pitch, yaw, roll;
}

// basic init method by setting the name of the object
- (id)initWithName:(NSString *)newName;

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary;

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary;

// sets the hitpoint
- (void)setHitPoint:(Vector3D *)newHitPoint;
// sets the object name... we had a few issues calling this method "setName" as polymorphically it interfered with some of the builtin classes. Not sure if we have fixed the problem now but it is simpler to leave it now.
- (void)setObjectName:(NSString *)newName;
// sets a new material object
- (void)setMaterial:(Material *)newMaterial;
// Sets the material by finding its name in the list of materials
- (void)setMaterialWithName:(NSString *)name materials:(NSArray *)materials;

// sets a new colour
- (void)setColour:(RGBVector *)newColour;

// returns hitpoint... the intersection point of a particular ray.
- (Vector3D *)hitPoint;
// returns the name
- (NSString *)name;
// returns a pointer to the material object of this object
- (Material *)material;
- (Transformation *)invTextureTransformation;
// returns a pointer to the colour
- (RGBVector *)diffuseColour:(Vector3D *)intersect;
- (float)diffuseReflection:(Vector3D *)intersect;
- (float)specularReflection:(Vector3D *)intersect;
- (BOOL)isDiffuseReflective:(Vector3D *)intersect;
- (BOOL)isSpecularReflective:(Vector3D *)intersect;
- (BOOL)isTransparent:(Vector3D *)intersect;
- (BOOL)isReflective:(Vector3D *)intersect;

// tests if theRay intersects with this object. This should be an abstract method and is not implemented here.
- (BOOL)intersects:(Ray *)theRay;

// returns the normal to the object at the given point. This should be an abstract method and is not implemented here.
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint;

//Debugging
- (void)logValues;

@end
