//
//  Object.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Object3D.h"


@implementation Object3D

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		hitPoint = [[Vector3D alloc] init];
		invTextureTransformation = [[Transformation alloc] initAsIdentity];
		name = [[NSMutableString alloc] init];
//		material = [[Material alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// basic init method by setting the name of the object
- (id)initWithName:(NSString *)newName
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// alloc and init all the stored objects
		hitPoint = [[Vector3D alloc] init];
		invTextureTransformation = [[Transformation alloc] initAsIdentity];
		name = [[NSMutableString alloc] initWithString:newName];
//		material = [[Material alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		int i;	//array enumerator
		id mat;
		hitPoint = [[Vector3D alloc] init];
		invTextureTransformation = [[Transformation alloc] initAsIdentity];
		//retrieve the name from the dict and pass it to the init for name
		name = [[NSMutableString alloc] initWithString:[dictionary valueForKey:@"Name"]];

		//retrieve the material dictionary from within the object dictionary and pass it to the material initWithDict method after checking the type of the material
		material = nil;
		//For every material in our materials array
		for (i=0; i<[[dictionary valueForKey:@"Materials"] count]; i++)
		{
			mat = [[dictionary valueForKey:@"Materials"] objectAtIndex:i];
			if ([mat isKindOfClass:[Material class]])
			{
				if ([[mat name] isEqualToString:[dictionary valueForKey:@"Material Name"]])
					material = mat;
			}
		}
		return self;
	}
	return nil; // something went wrong.
}

- (void)dealloc
{
	// dealloc memory created by this class
	[name release];
	[hitPoint release];
	[invTextureTransformation release];
//	[material release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSMutableDictionary *)asDictionary
{
	return [NSMutableDictionary dictionaryWithObjects:
				[NSArray arrayWithObjects:
					// grab the material object as a dictionary
					[material name],
			//		[NSDictionary dictionaryWithDictionary:[material asDictionary]],
					// grab name
					name,
					// tell the dictionary what type of object this is, hopefully no-one will try to instantiate this class
					@"Abstract Object",
					nil
				]
				// give the above objects keys in the dictionary
				forKeys:[NSArray arrayWithObjects:@"Material Name", @"Name", @"Object Type", nil]
			];
}

// returns hitpoint... the intersection point of a particular ray.
- (Vector3D *)hitPoint						{	return hitPoint;			}
// returns the name
- (NSString *)name							{	return name;				}
// sets a new colour
- (void)setColour:(RGBVector *)newColour		{	[material setDiffuseColour:newColour];	}
// sets a new material object
- (void)setMaterial:(Material *)newMaterial		{	material = newMaterial;	}

// Sets the material by finding its name in the list of materials
- (void)setMaterialWithName:(NSString *)newName materials:(NSArray *)materials
{
	int i;
	id mat;
	
	// Set the current material of this object to nil
	material = nil;

	//For every material in our materials array
	for (i=0; i<[materials count]; i++)
	{
		mat = [materials objectAtIndex:i];
		if ([mat isKindOfClass:[Material class]])
		{
			if ([[mat name] isEqualToString:newName])
				material = mat;
		}
	}
	
	// If we have not found the material name in the array, set the material to the default material
	if (material == nil)
		material = [materials objectAtIndex:0];
}

// sets the hitpoint
- (void)setHitPoint:(Vector3D *)newHitPoint	{	[hitPoint assign:newHitPoint];	}
// sets the object name... we had a few issues calling this method "setName" as polymorphically it interfered with some of the builtin classes. Not sure if we have fixed the problem now but it is simpler to leave it now.
- (void)setObjectName:(NSString *)newName	{	[name setString:newName];	}
// returns a pointer to the material object of this object
- (Material *)material							{	return material;				}
- (Transformation *)invTextureTransformation	{	return invTextureTransformation;	}
// It may seem silly to do these here, but they come in useful when we apply textures to generic objects, so we need these methods here for the inheritance. We
- (RGBVector *)diffuseColour:(Vector3D *)intersect	{	return [material diffuseColour:intersect];		}
- (float)diffuseReflection:(Vector3D *)intersect	{	return [material diffuseReflection:intersect];	}
- (float)specularReflection:(Vector3D *)intersect	{	return [material specularReflection:intersect];	}
- (BOOL)isDiffuseReflective:(Vector3D *)intersect	{	return [material isDiffuseReflective:intersect];	}
- (BOOL)isSpecularReflective:(Vector3D *)intersect	{	return [material isSpecularReflective:intersect];	}
- (BOOL)isTransparent:(Vector3D *)intersect			{	return [material isTransparent:intersect];		}
- (BOOL)isReflective:(Vector3D *)intersect			{	return [material isReflective:intersect];		}

// tests if theRay intersects with this object. This should be an abstract method and is not implemented here.
- (BOOL)intersects:(Ray *)theRay	{	return NO;	}	// return NO because this method cannot be implemented here.

// This is just a temporary measure. We will make this method abstract (and most of the class)
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint 	{	return hitPoint;	}

//Debugging
- (void)logValues
{
	NSLog([@"Name: " stringByAppendingString:name]);
}

@end
