//
//  Plane.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Tue Aug 14 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Plane.h"


@implementation Plane
// Class methods
+ (Plane *)planeWithDictionary:(NSDictionary *)dictionary
{	return [[[Plane alloc] initWithDictionary:dictionary] autorelease];	}

// Boring init
- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		point = [[Vector3D alloc] init];
		normal = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// Initialisation
- (id)initWithVectorsPoint:(Vector3D *)newPoint normal:(Vector3D *)newNormal name:(NSString *)newName
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		[point assign:newPoint];
		[normal assign:newNormal];
		[name setString:newName];
		return self;
	}
	return nil;	//	something went wrong.
}

// Initialisation
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		point = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Point"]];
		normal = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Normal"]];
		return self;
	}
	return nil; // something went wrong.
}

// Copy constructor
- (id)copyWithZone:(NSZone *)zone
{	return [[Plane allocWithZone:zone] initWithVectorsPoint:point normal:normal name:name];		}

// Destructor
- (void)dealloc
{
	[point release];
	[normal release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// Convert the object to a dictionary
- (NSDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[point asArray] forKey:@"Point"];
	[dictionary setObject:[normal asArray] forKey:@"Normal"];
	[dictionary setObject:@"Plane" forKey:@"Object Type"];
	return dictionary;
}

// Getters
- (Vector3D *)point		 	{ 	return point;		}
- (Vector3D *)normal		{	return normal;		}

// Setters
- (void)setPointWithVector:(Vector3D *)newPoint							{	[point assign:newPoint];					}
- (void)setPointWithValuesX:(float)newX y:(float)newY z:(float)newZ		{	[point setWithValuesX:newX y:newY z:newZ];		}
- (void)setNormalWithVector:(Vector3D *)newNormal						{	[normal assign:newNormal];					}
- (void)setNormalWithValuesX:(float)newX y:(float)newY z:(float)newZ	{	[normal setWithValuesX:newX y:newY z:newZ];		}

// Intersection routine for the plane. Much easier than for a sphere
- (BOOL)intersects:(Ray *)theRay
{
	float t;	// Ray parameter
	
	t = ([[point subtractVector:[theRay origin]] dotProduct:normal]) / ([[theRay direction] dotProduct:normal]);
	
	if (t > EPSILON)	// To avoid salt and pepper
	{
		[hitPoint assign:[[theRay origin] addVector:[[theRay direction] multiplyByFloat:t]]];
		[theRay setT:t];
		return YES;
	}
	return NO;
}

// Easiest normalAtPoint ever
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint					{	return [normal unit];	}

@end
