//
//  RGBVector.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Sat Aug 11 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "RGBVector.h"


@implementation RGBVector

// Class methods for allocating, basically syntactical sugar, the same thing can be achieved by calling alloc and then initWithValues. See notes
+ (RGBVector *)rgbVectorWithRGBVector:(RGBVector *)rhs
{	return [[[self alloc] initWithRGBVector:rhs] autorelease];	}
+ (RGBVector *)rgbVectorWithValue:(float)newValue
{	return [[[self alloc] initWithValuesRed:newValue green:newValue blue:newValue] autorelease];	}
+ (RGBVector *)rgbVectorWithValuesRed:(float)newRed green:(float)newGreen blue:(float)newBlue
{	return [[[self alloc] initWithValuesRed:newRed green:newGreen blue:newBlue] autorelease];		}
+ (RGBVector *)rgbVectorWithArray:(NSArray *)array
{	return [[[self alloc] initWithArray:array] autorelease];	}

+ (RGBVector *)rgbVectorWithLinearFrameStart:(RGBVector *)start end:(RGBVector *)end currentFrame:(int)currentFrame totalFrames:(int)totalFrames
{
	return [RGBVector rgbVectorWithValuesRed:[start red] + (([end red] - [start red])/(totalFrames - 1))*currentFrame
										green:[start green] + (([end green] - [start green])/(totalFrames - 1))*currentFrame
										blue:[start blue] + (([end blue] - [start blue])/(totalFrames - 1))*currentFrame];
}

+ (RGBVector *)rgbVectorInRamp:(RGBVector *)start end:(RGBVector *)end position:(float)position
{
	return [RGBVector rgbVectorWithValuesRed:[start red]	+ ([end red] - [start red]) * position
										green:[start green]	+ ([end green] - [start green]) * position
										blue:[start blue]	+ ([end blue] - [start blue]) * position];
}

// Initiallise with the same value for each channel
- (id)initWithValue:(float)newValue		{	return [self initWithValuesRed:newValue green:newValue blue:newValue];	}

// Initiallise with separate values
- (id)initWithValuesRed:(float)newRed green:(float)newGreen blue:(float)newBlue
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		red = newRed;
		green = newGreen;
		blue = newBlue;
		return self;
	}
	return nil;	// Something went wrong
}

// Initialise with another RGBVector. This may seem silly, and it probably is, but hey we are doing it this way and there is nothing you can do about it. So ner
- (id)initWithRGBVector:(RGBVector *)newRGBVector
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		red = [newRGBVector red];
		green = [newRGBVector green];
		blue = [newRGBVector blue];
		return self;
	}
	return nil;	// Something went wrong
}

- (id)initWithArray:(NSArray *)array;
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		red = [[array objectAtIndex:0] floatValue];
		green = [[array objectAtIndex:1] floatValue];;
		blue = [[array objectAtIndex:2] floatValue];;
		return self;
	}
	return nil;	// Something went wrong
}

- (NSArray *)asArray
{
	return [NSArray arrayWithObjects:
			[NSNumber numberWithFloat:red],
			[NSNumber numberWithFloat:green],
			[NSNumber numberWithFloat:blue],
			nil];
}

// Assignment method
- (void)assign:(RGBVector *)rhs
{
	red = [rhs red];
	green = [rhs green];
	blue = [rhs blue];
}

// Setters...
- (void)setWithValuesRed:(float)newRed green:(float)newGreen blue:(float)newBlue
{
	red = newRed;
	green = newGreen;
	blue = newBlue;
}

- (void)setWithValuesNSColor:(NSColor *)newColour 	{	[newColour getRed:&red green:&green blue:&blue alpha:nil];	}

- (void)setRed:(float)newRed		{	red = newRed;		}
- (void)setGreen:(float)newGreen	{	green = newGreen;	}
- (void)setBlue:(float)newBlue		{	blue = newBlue;		}
// Getters...
- (float)red						{	return red;		}
- (float)green						{	return green;	}
- (float)blue						{	return blue;	}

// Makes sure all colour channels are 0 <= x <= 1
- (void)clamp
{
	// We should never need the < 0, but this is just for Tim's pedanticness. Stuart objected (speed?), over-ruled. We don't call this very often (Note: Tim is typing this...)
	red = (red > 1) ? 1 : red;
	green = (green > 1) ? 1 : green;
	blue = (blue > 1) ? 1 : blue;
	red = (red < 0) ? 0 : red;
	green = (green < 0) ? 0 : green;
	blue = (blue < 0) ? 0 : blue;
}

// Mathematical manipulation of colours
- (RGBVector *)multiplyByFloat:(float)rhs
{	return [RGBVector rgbVectorWithValuesRed:(red * rhs) green:(green * rhs) blue:(blue * rhs)];	}
- (RGBVector *)multiplyByColour:(RGBVector *)rhs
{	return [RGBVector rgbVectorWithValuesRed:(red * [rhs red]) green:(green * [rhs green]) blue:(blue * [rhs blue])];	}
- (RGBVector *)addColour:(RGBVector *)rhs
{	return [RGBVector rgbVectorWithValuesRed:(red + [rhs red]) green:(green + [rhs green]) blue:(blue + [rhs blue])];	}
- (RGBVector *)averageColour:(RGBVector *)rhs
{	return [RGBVector rgbVectorWithValuesRed:((red + [rhs red])/2) green:((green + [rhs green])/2) blue:((blue + [rhs blue])/2)];	}
- (RGBVector *)attenuateColour:(RGBVector *)rhs depth:(int)depth
{	return [RGBVector rgbVectorWithValuesRed:((red + [rhs red])/depth) green:((green + [rhs green])/depth) blue:((blue + [rhs blue])/depth)];	}
// Returns YES if any absolute colour difference is greater than the tolerance
- (BOOL)toleranceCompare:(RGBVector *)rhs tolerance:(float)tolerance
{	return (fabs([rhs red] - red) > tolerance || fabs([rhs green] - green) > tolerance || fabs([rhs blue] - blue) > tolerance);	}

// Debugging
- (void)logValues
{	NSLog(@"Red: %f, Green: %f, Blue: %f", red, green, blue);	}
@end
