//
//  SceneParser.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Sun Aug 19 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "SceneParser.h"

@implementation SceneParser

// Main import scene function, calls the three below
- (BOOL)importSceneWithCamera:(Camera *)camera materials:(NSMutableArray *)materials objects:(NSMutableArray *)objects lights:(NSMutableArray *)lights
						ambient:(RGBVector *)ambient background:(RGBVector *)background filename:(NSString *)filename
{
	NSLog(filename);
	importedScene = [[NSDictionary alloc] initWithContentsOfFile:filename];
	
	// Something went wrong reading the file
	if (importedScene == nil)
		return NO;
		
	if (![self validateScene])
		return NO;
		
	[self importCamera:camera];
	[self importMaterials:materials];
	[self importObjects:objects materials:materials];
	[self importLights:lights];

	// Since ambient is so small we just do it here
	[ambient setWithValuesRed:[[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:0] floatValue]
		green:[[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:1] floatValue]
		blue:[[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:2] floatValue]
		];
	// Same with background colour
	[background setWithValuesRed:[[[importedScene valueForKey:@"Background Colour"] objectAtIndex:0] floatValue]
		green:[[[importedScene valueForKey:@"Background Colour"] objectAtIndex:1] floatValue]
		blue:[[[importedScene valueForKey:@"Background Colour"] objectAtIndex:2] floatValue]
		];

	[importedScene release];
	return YES;
}

// Import the camera data from a dictionary
- (void)importCamera:(Camera *)camera
{	[camera initWithDictionary:[importedScene valueForKey: @"Camera"]];		}

// Import the material data from a dictionary
- (void)importMaterials:(NSMutableArray *)materials
{
	NSMutableDictionary *tempDictionary;
	int i;
	if ([materials count] > 0)
	{
        if ([[materials objectAtIndex:[materials count]-1] isKindOfClass:[NSString class]])
        {
            [materials removeObjectAtIndex:[materials count]-1];
        }
	}
	for (i = 0; i <= [[importedScene valueForKey: @"Materials"] count] - 1; i++)
	{
		tempDictionary = [[NSDictionary alloc] initWithDictionary:[[importedScene valueForKey: @"Materials"] objectAtIndex:i]];
		// Each material type knows how to read its dictionary
		
		if ([[tempDictionary valueForKey:@"Material Type"] isEqualToString:@"Sphere Mapping"])
			if ([[tempDictionary valueForKey:@"Name"] isEqualToString:@"Default Material"])
				[materials replaceObjectAtIndex:0 withObject:[SphereMapping sphereMappingWithDictionary:tempDictionary]];
			else
				[materials addObject:[SphereMapping sphereMappingWithDictionary:tempDictionary]];

		else if ([[tempDictionary valueForKey:@"Material Type"] isEqualToString:@"Marble"])
			if ([[tempDictionary valueForKey:@"Name"] isEqualToString:@"Default Material"])
				if ([materials count] > 0)
					[materials replaceObjectAtIndex:0 withObject:[Marble marbleWithDictionary:tempDictionary]];
				else
					[materials addObject:[Marble marbleWithDictionary:tempDictionary]];
			else
				[materials addObject:[Marble marbleWithDictionary:tempDictionary]];

		else if ([[tempDictionary valueForKey:@"Material Type"] isEqualToString:@"Phong"])
			if ([[tempDictionary valueForKey:@"Name"] isEqualToString:@"Default Material"])
				if ([materials count] > 0)
					[materials replaceObjectAtIndex:0 withObject:[Material materialWithDictionary:tempDictionary]];
				else
					[materials addObject:[Material materialWithDictionary:tempDictionary]];
			else
				[materials addObject:[Material materialWithDictionary:tempDictionary]];
		[tempDictionary release];
	}

	// We actually need to loop through the array twice to make sure that the materials required by checkerboard are already in the array
	for (i = 0; i <= [[importedScene valueForKey: @"Materials"] count] - 1; i++)
	{
		tempDictionary = [[NSMutableDictionary alloc] initWithDictionary:[[importedScene valueForKey: @"Materials"] objectAtIndex:i]];
		[tempDictionary setObject:materials forKey:@"Materials"];

		if ([[tempDictionary valueForKey:@"Material Type"] isEqualToString:@"Checkerboard"])
			if ([[tempDictionary valueForKey:@"Name"] isEqualToString:@"Default Material"])
				[materials replaceObjectAtIndex:0 withObject:[Checkerboard checkerboardWithDictionary:tempDictionary]];
			else
				[materials addObject:[Checkerboard checkerboardWithDictionary:tempDictionary]];

		[tempDictionary release];
	}

	// This is used by NSOutlineView to display a name for the container
        [materials addObject:@"Materials"];
}

//This is designed to be able to import just materials whether or not a scene is presently loaded
- (BOOL)importMaterialLibrary:(NSMutableArray *)materials filename:(NSString *)filename
{
    NSLog(filename);
    importedScene = [[NSDictionary alloc] initWithContentsOfFile:filename];
    if (importedScene == nil)
        return NO;
    [self importMaterials:materials];
    [importedScene release];
    return YES;
}


// Import the object data from a dictionary
- (void)importObjects:(NSMutableArray *)threeDObjects materials:(NSMutableArray *)materials
{
	NSMutableDictionary *tempDictionary;
	int i;
	for (i = 0; i <= [[importedScene valueForKey: @"Objects"] count] - 1; i++)
	{
		tempDictionary = [[NSMutableDictionary alloc] initWithDictionary:[[importedScene valueForKey: @"Objects"] objectAtIndex:i]];
		[tempDictionary setObject:materials forKey:@"Materials"];
		// I would really like a way to dynamically choose the object type instead of adding to this every time we add a new object type, but we can't think of a way to do this. PS There must be a way!!! Stuart is sceptical, Tim is dreaming
		// Each object type knows how to read its dictionary
		if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Sphere"])
			[threeDObjects addObject:[Sphere sphereWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Plane"])
			[threeDObjects addObject:[Plane planeWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Triangle"])
			[threeDObjects addObject:[Triangle triangleWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Generic Box"])
			[threeDObjects addObject:[GenericBox genericBoxWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Generic Sphere"])
			[threeDObjects addObject:[GenericSphere genericSphereWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Generic Torus"])
			[threeDObjects addObject:[GenericTorus genericTorusWithDictionary:tempDictionary]];
		else if ([[tempDictionary valueForKey:@"Object Type"] isEqualToString:@"Generic Tapered Cylinder"])
			[threeDObjects addObject:[GenericTaperedCylinder genericTaperedCylinderWithDictionary:tempDictionary]];

		[tempDictionary release];
	}
	// This is used by NSOutlineView to display a name for the container
	[threeDObjects addObject:@"3D Objects"];
}

// Import the light data from a dictionary
- (void)importLights:(NSMutableArray *)lights
{
	NSDictionary *tempDictionary;
	Light *tempLight;
	int i;
	for (i = 0; i <= [[importedScene valueForKey: @"Lights"] count] - 1; i++)
	{
		// This is basically the same as for importObjects
		tempDictionary = [[NSDictionary alloc] initWithDictionary:[[importedScene valueForKey: @"Lights"] objectAtIndex:i]];
		if ([[tempDictionary valueForKey:@"Light Type"] isEqualToString:@"Point"])
		{
			tempLight = [[Light alloc] initWithDictionary:tempDictionary];
			[lights addObject:tempLight];
			[tempLight release];
		}
		[tempDictionary release];
	}
	[lights addObject:@"Lights"];
}

// Main save scene function, calls the three below
- (BOOL)saveSceneWithCamera:(Camera *)camera materials:(NSMutableArray *)materials objects:(NSMutableArray *)threeDObjects lights:(NSMutableArray *)lights
					ambient:(RGBVector *)ambient background:(RGBVector *)background filename:(NSString *)filename 
{
	exportedScene = [[NSMutableDictionary alloc] init];
	[self saveCamera:camera];
	[self saveMaterials:materials];
	[self saveObjects:threeDObjects];
	[self saveLights:lights];
		
	// Since ambient light is small, we write it out here
	[exportedScene setObject:[NSArray arrayWithObjects:
		[NSNumber numberWithFloat:[ambient red]],
		[NSNumber numberWithFloat:[ambient green]],
		[NSNumber numberWithFloat:[ambient blue]],
 		nil
	] forKey:@"Ambient Light"];

	// Same with background colour
	[exportedScene setObject:[NSArray arrayWithObjects:
		[NSNumber numberWithFloat:[background red]],
		[NSNumber numberWithFloat:[background green]],
		[NSNumber numberWithFloat:[background blue]],
 		nil
	] forKey:@"Background Colour"];
	[exportedScene writeToFile:filename atomically:YES];
	NSLog(filename);
	[exportedScene release];
	return YES;
}

// Convert the camera data to a dictionary
- (void)saveCamera:(Camera *)camera
{
	// How hard would it be to write asDictionary for camera?? Why don't we bother?? Oh yeah, too lazy. Hey, what do you know, we did get around to it
	[exportedScene setObject:[camera asDictionary] forKey:@"Camera"];
}

// Convert the material data to a dictionary
- (void)saveMaterials:(NSMutableArray *)materials
{
	NSMutableArray *materialsAsDictionaries = [[NSMutableArray alloc] init];
	NSEnumerator *en;
	Material *material;

	// For every object we call asDictionary and add that to our exportScene dictionary
	for (en=[materials objectEnumerator]; material=[en nextObject];)
		if ([material isKindOfClass:[Material class]])
			[materialsAsDictionaries addObject:[material asDictionary]];

	[exportedScene setObject:materialsAsDictionaries forKey:@"Materials"];
}

// Convert the object data to a dictionary
- (void)saveObjects:(NSMutableArray *)threeDObjects
{
	NSMutableArray *objectsAsDictionaries = [[NSMutableArray alloc] init];
	NSEnumerator *en;
	Object3D *obj;

	// For every object we call asDictionary and add that to our exportScene dictionary
	for (en=[threeDObjects objectEnumerator]; obj=[en nextObject];)
		if ([obj isKindOfClass:[Object3D class]])
			[objectsAsDictionaries addObject:[obj asDictionary]];

	[exportedScene setObject:objectsAsDictionaries forKey:@"Objects"];
}

// Convert the light data to a dictionary
- (void)saveLights:(NSMutableArray *)lights
{
	NSMutableArray *lightsAsDictionaries = [[NSMutableArray alloc] init];
	NSEnumerator *en;
	Light *obj;

	// For every light we call asDictionary and add that to our exportScene dictionary
	for (en=[lights objectEnumerator]; obj=[en nextObject];)
		if ([obj isKindOfClass:[Light class]])
			[lightsAsDictionaries addObject:[obj asDictionary]];

	[exportedScene setObject:lightsAsDictionaries forKey:@"Lights"];
}

// Validate the scene file that is being imported
- (BOOL)validateScene
{
	return ([self validateAmbientLight] && [self validateBackgroundColour] && [self validateCamera]);
}

- (BOOL)validateAmbientLight
{
	//Validate Ambient Lighting
	if ([importedScene valueForKey:@"Ambient Light"] == nil)
	{
		NSLog(@"Ambient Light not present.");
		return NO;
	}
	if (![[importedScene valueForKey:@"Ambient Light"] isKindOfClass:[NSArray class]])
	{
		NSLog(@"Ambient Light not of class Array.");
		return NO;
	}
	if ([[importedScene valueForKey:@"Ambient Light"] count] < 3
		|| [[importedScene valueForKey:@"Ambient Light"] count] > 3)
	{
		NSLog(@"Ambient Light has too many or too few elements in array.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:0] floatValue] > 1
		|| [[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:0] floatValue] < 0)
	{
		NSLog(@"Ambient Light Red not in range 0-1.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:1] floatValue] > 1
		|| [[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:1] floatValue] < 0)
	{
		NSLog(@"Ambient Light Green not in range 0-1.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:2] floatValue] > 1
		|| [[[importedScene valueForKey:@"Ambient Light"] objectAtIndex:2] floatValue] < 0)
	{
		NSLog(@"Ambient Light Blue not in range 0-1.");
		return NO;
	}
	return YES;
}

- (BOOL)validateBackgroundColour
{
	//Validate Ambient Lighting
	if ([importedScene valueForKey:@"Background Colour"] == nil)
	{
		NSLog(@"Background Colour not present.");
		return NO;
	}
	if (![[importedScene valueForKey:@"Background Colour"] isKindOfClass:[NSArray class]])
	{
		NSLog(@"Background Colour not of class Array.");
		return NO;
	}
	if ([[importedScene valueForKey:@"Background Colour"] count] < 3
		|| [[importedScene valueForKey:@"Background Colour"] count] > 3)
	{
		NSLog(@"Background Colour has too many or too few elements in array.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Background Colour"] objectAtIndex:0] floatValue] > 1
		|| [[[importedScene valueForKey:@"Background Colour"] objectAtIndex:0] floatValue] < 0)
	{
		NSLog(@"Background Colour Red not in range 0-1.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Background Colour"] objectAtIndex:1] floatValue] > 1
		|| [[[importedScene valueForKey:@"Background Colour"] objectAtIndex:1] floatValue] < 0)
	{
		NSLog(@"Background Colour Green not in range 0-1.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Background Colour"] objectAtIndex:2] floatValue] > 1
		|| [[[importedScene valueForKey:@"Background Colour"] objectAtIndex:2] floatValue] < 0)
	{
		NSLog(@"Background Colour Blue not in range 0-1.");
		return NO;
	}
	return YES;
}

- (BOOL)validateCamera
{
	//Validate camera
	if ([importedScene valueForKey: @"Camera"] == nil)
	{
		NSLog(@"Camera not present.");
		return NO;
	}
	if (![[importedScene valueForKey:@"Camera"] isKindOfClass:[NSDictionary class]])
	{
		NSLog(@"Camera not of class Dictionary.");
		return NO;
	}
	
	//Validate camera location
	if ([[importedScene valueForKey:@"Camera"] valueForKey:@"Location"] == nil)
	{
		NSLog(@"Camera Location not present.");
		return NO;
	}
	if (![[[importedScene valueForKey:@"Camera"] valueForKey:@"Location"] isKindOfClass:[NSArray class]])
	{
		NSLog(@"Camera Location not of class Array.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Camera"] valueForKey:@"Location"] count] < 3
		|| [[[importedScene valueForKey:@"Camera"] valueForKey:@"Location"] count] > 3)
	{
		NSLog(@"Camera Location has too many or too few elements in array.");
		return NO;
	}
	
	//Validate camera lookAt
	if ([[importedScene valueForKey:@"Camera"] valueForKey:@"Look At"] == nil)
	{
		NSLog(@"Camera Look At not present.");
		return NO;
	}
	if (![[[importedScene valueForKey:@"Camera"] valueForKey:@"Look At"] isKindOfClass:[NSArray class]])
	{
		NSLog(@"Camera Look At not of class Array.");
		return NO;
	}
	if ([[[importedScene valueForKey:@"Camera"] valueForKey:@"Look At"] count] < 3
		|| [[[importedScene valueForKey:@"Camera"] valueForKey:@"Look At"] count] > 3)
	{
		NSLog(@"Camera Look At has too many or too few elements in array.");
		return NO;
	}
	if ([[importedScene valueForKey:@"Camera"] valueForKey:@"View Plane Distance"] == nil)
	{
		NSLog(@"Camera View Plane Distance not present.");
		return NO;
	}
	if ([[importedScene valueForKey:@"Camera"] valueForKey:@"Twist"] == nil)
	{
		NSLog(@"Camera Twist not present.");
		return NO;
	}
	return YES;
}


@end
