//
//  ShadingObject.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 17 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "ShadingObject.h"

@implementation ShadingObject
// Boring init
- (id)init
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		intersection = [[Vector3D alloc] init];
		transIntersection = [[Vector3D alloc] init];
		normal = [[Vector3D alloc] init];
		ray = [[Ray alloc] init];
		inside = NO;
		return self;
	}
	return nil; // something went wrong.
}

// Initialise with everything anyone would ever need
- (id)initWithValuesObjectPtr:(Object3D *)objectPtr intersection:(Vector3D *)newIntersection normal:(Vector3D *)newNormal ray:(Ray *)newRay
					ambient:(RGBVector *)newAmbient
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
//		object = objectPtr;
		intersection = [[Vector3D alloc] initWithVector:newIntersection];
		transIntersection = [[Vector3D alloc] initWithVector:[intersection transformAsPoint:[objectPtr invTextureTransformation]]];
		normal = [[Vector3D alloc] initWithVector:newNormal];
		ray = [[Ray alloc] initWithRay:newRay];
		inside = NO;
		return self;
	}
	return nil; // something went wrong.
}

// Release all memory
- (void)dealloc
{
	[intersection release];
	[transIntersection release];
	[normal release];
	[ray release];
	[super dealloc];  //use the super class's method for instance destruction
}

// Getters
- (Object3D *)object		{	return object;			}
- (Vector3D *)intersection	{	return intersection;	}
- (Vector3D *)transIntersection		{	return transIntersection;	}
- (Vector3D *)normal		{	return normal;			}
- (Ray *)ray				{	return ray;				}
- (BOOL)killed				{	return killed;			}
- (BOOL)inside				{	return inside;			}

// Setters
- (void)setObject:(Object3D *)newObject					{	object = newObject;		}
- (void)setNormal:(Vector3D *)newNormal					{	[normal assign:newNormal];	}
- (void)setIntersection:(Vector3D *)newIntersection
{
	[intersection assign:newIntersection];
	[transIntersection assign:[intersection transformAsPoint:[object invTextureTransformation]]];
//	NSLog([object name]);
//	[transIntersection logValues];
}
- (void)setRay:(Ray *)newRay							{	[ray assign:newRay];	}
- (void)setLights:(NSMutableArray *)newLights			{	lights = newLights;		}
- (void)setObjects:(NSMutableArray *)newObjects			{	objects = newObjects;	}
- (void)setCameraLocation:(Vector3D *)newCameraLocation	{	cameraLocation = newCameraLocation;		}
- (void)setAmbientLight:(RGBVector *)newAmbientLight	{	ambient = newAmbientLight;				}
- (void)setKilled:(BOOL)newKilled						{	killed = newKilled;		}
- (void)setInside:(BOOL)newInside						{	inside = newInside;		}

// Colour getters
- (RGBVector *)diffuseColour	{	return [[object material] diffuseColour:transIntersection];		}
- (BOOL)isReflective			{	return [[object material] isReflective:transIntersection];		}
- (BOOL)isTransparent			{	return [[object material] isTransparent:transIntersection];		}
- (float)globalReflection		{	return [[object material] globalReflection:transIntersection];	}
- (float)globalTransmission		{	return [[object material] globalTransmission:transIntersection];	}

- (RGBVector *)attenuationColour
{
	RGBVector *attenuation = [RGBVector rgbVectorWithRGBVector:[self diffuseColour]];

	[attenuation setRed:(pow([attenuation red], [ray t]))];
	[attenuation setGreen:(pow([attenuation green], [ray t]))];
	[attenuation setBlue:(pow([attenuation blue], [ray t]))];
	return attenuation;
}

// Basic phong shading formula
- (RGBVector *)localColour
{
	RGBVector *diffuseColour = [[object material] diffuseColour:transIntersection];
	return [[diffuseColour multiplyByColour:ambient] addColour:[diffuseColour multiplyByColour:[self calculateIntensity]]];
}

// Calculate the intensity of all the light sources shining on this object
- (RGBVector *)calculateIntensity
{
	BOOL inShadow = NO;
	// We start out with the intensity at black, doesn't just specify brightness, but also colour
	RGBVector *intensity = [RGBVector rgbVectorWithValuesRed:0.0 green:0.0 blue:0.0],
				*diffuseIntensity = [[RGBVector alloc] init],
				*specularIntensity = [[RGBVector alloc] init];
	// Used to iterate through the arrays
	NSEnumerator *lightsEn, *objEn;
	Light *currentLight;
	Object3D *currentObject;
	Ray *currentRay = [[Ray alloc] init];
	
	// Must calculate the direction of the ray from the intersection point stored above to each point light source.
	// Enumerate through every light
	for (lightsEn=[lights objectEnumerator]; currentLight=[lightsEn nextObject];)
	{
		//Make sure the object is a light because the array will store a string at the end for use by NSOutlineView
		if ([currentLight isKindOfClass:[Light class]])
		{
			// Set the direction and origin of the current shadow feeler ray
			[currentRay setDirectionWithVector:[[[currentLight location] subtractVector:intersection] unit]];
			[currentRay setOriginWithVector:intersection];
			// Enumerate through every object
			for (objEn=[objects objectEnumerator]; currentObject=[objEn nextObject];)
			{
				// Make sure the object is of type Object3D. The array will store a string at the end for use by NSOutlineView
				if ([currentObject isKindOfClass:[Object3D class]])
				{
					// Check if the current light intersects with the current object
					// The second test eliminates incorrect shadows, ie a shadows where the object is not between the intersection point and the light	
					if ([currentObject intersects:currentRay] && [currentRay t] < [[currentRay direction] length])
						inShadow = YES;
				}
				// This object is in shadow, no need to test any other objects, break from for (objects) loop
				if (inShadow) break;
			}	// end object enumeration

			// The object is not in shadow, so we actually need to shade it!!
			if (!inShadow)
			{
				float cosTheta = [normal dotProduct:[currentRay direction]];
				// This side of the object is not facing the light if cosTheta is < 0
				if (cosTheta > 0)
				{
					Vector3D *cameraDirection = [[Vector3D alloc] initWithVector:[[cameraLocation subtractVector:intersection] unit]];
					Vector3D *reflectedLightRay = [[Vector3D alloc] init];

					// Find the direction of the reflected ray
					[reflectedLightRay assign:[[[currentRay direction] subtractVector:[normal multiplyByFloat:2*cosTheta]] unit]];
					// Assign diffuseIntensity according to phong formula
					[diffuseIntensity assign:[[currentLight colourWithIntensity]
									multiplyByFloat:([[object material] diffuseReflection:transIntersection] *  cosTheta)]];
					// Assign specularIntensity according to phong formula
					[specularIntensity assign:[[currentLight colourWithIntensity] multiplyByFloat:([[object material] specularReflection:transIntersection] *
						pow([reflectedLightRay dotProduct:cameraDirection], [[object material] specularExponent:transIntersection]) )]];

					// Add on the calculated colours
					[intensity assign:[intensity addColour:diffuseIntensity]];
					[intensity assign:[intensity addColour:specularIntensity]];
					// Release objects assigned in this if
					[cameraDirection release];
					[reflectedLightRay release];
				}
			}
			inShadow = NO;
		}
	}	//end light enumeration
	//Clamp the intensity
	[intensity clamp];
	// Release memory
	[currentRay release];
	[diffuseIntensity release];
	[specularIntensity release];
	return intensity;
}

@end
