//
//  Sphere.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Sphere.h"


@implementation Sphere
// Class methods
+ (Sphere *)sphereWithDictionary:(NSDictionary *)dictionary
{	return [[[Sphere alloc] initWithDictionary:dictionary] autorelease];	}

// For making movies
+ (Sphere *)sphereWithLinearFrameStart:(Sphere *)start end:(Sphere *)end currentFrame:(int)currentFrame totalFrames:(int)totalFrames
{
	Sphere *tempSphere = [[[Sphere alloc] init] autorelease];
	[tempSphere setCentre:[Vector3D vector3DWithLinearFrameStart:[start centre] end:[end centre] currentFrame:currentFrame totalFrames:totalFrames]];
	[tempSphere setRadius:[start radius] + (([end radius] - [start radius])/(totalFrames - 1))*currentFrame];
	return tempSphere;
}

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// allocate memory for the centre
		centre = [[Vector3D alloc] init];
		radius = 1;	// set radius to a default
		return self;
	}
	return nil; // something went wrong.
}

// basic init method by setting the name of the object
- (id)initWithName:(NSString *)newName
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// allocate memory for the centre
		centre = [[Vector3D alloc] init];
		radius = 1;	// set radius to a default
		// set the name of the object. Name has been allocated in the super Object3D.
		[name setString:newName];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		// Grab the centre from the dictionary and alloc memory to store it in this object
		centre = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Centre"]];
		//Grab the radius
		radius = [[dictionary valueForKey:@"Radius"] floatValue];
		return self;
	}
	return nil; // something went wrong.
}

// copyWithZone is needed for use by NSOutlineView which is used in our editor drawer. All classes which appear in the NSOutlineView will implement this method.
- (id)copyWithZone:(NSZone *)zone	{	return [[Sphere allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

- (void)dealloc
{
	//release threeDObjects array, dealloc is called when the object is going to be destroyed
	[centre release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[centre asArray] forKey:@"Centre"];
	[dictionary setObject:[NSNumber numberWithFloat:radius]	forKey:@"Radius"];
	[dictionary setObject:@"Sphere" forKey:@"Object Type"];
	return dictionary;
}

// returns a pointer to the centre
- (Vector3D *)centre	{	return centre;	}
// returns the radius
- (float)radius			{	return radius;	}

// sets the sphere with all data relevant to a sphere. This is generally not needed since we have the dictionary
- (void)setValuesX:(float)newX y:(float)newY z:(float)newZ radius:(float)newRadius name:(NSString *)newName
{
	[centre setWithValuesX:newX y:newY z:newZ];
	radius = newRadius;
	[name setString:newName];
}

// sets the centre
- (void)setCentre:(Vector3D *)newPoint	{	[centre assign:newPoint];	}
// sets the radius
- (void)setRadius:(float)newRadius		{	radius = newRadius;			}

// implementation of the abstract method in Object3D. This returns true if theRay intersects with the Sphere
- (BOOL)intersects:(Ray *)theRay
{
	// declare local variables and init them to the discriminant equation
	float t, a, b, c, discrim, e, denom;
	Vector3D *temp = [[Vector3D alloc] init];
	[temp assign:[[theRay origin] subtractVector:centre]];
	a = [[theRay direction] dotProduct:[theRay direction]];
	b = [[temp multiplyByFloat:2.0] dotProduct:[theRay direction]];
	c = [temp dotProduct:temp] - radius * radius;
	discrim = b * b - 4.0 * a * c;

	// This vector has done its duty and can now be set free to roam the savanna
	[temp release];

	// if discrim is < 0 then the ray did not intersect
	if (discrim < 0)
		return NO;
	else
	{
		e = sqrt(discrim);
		denom = 2.0 * a;
		t = (-b - e) / denom;	// Smaller root
		
		if (t > EPSILON)
		{
			// The ray did intersect and we are storing the intersect point
			[hitPoint assign:[[theRay origin] addVector:[[theRay direction] multiplyByFloat:t]]];
			// set the distance of the ray
			[theRay setT:t];
			return YES;
		}
		
		t = (-b + e) / denom;	// Larger root
		
		if (t > EPSILON)
		{
			// The ray did intersect and we are storing the intersect point
			[hitPoint assign:[[theRay origin] addVector:[[theRay direction] multiplyByFloat:t]]];
			// set the distance of the ray
			[theRay setT:t];
			return YES;
		}
	}

	return NO;
}

// implementation of the abstract method in Object3D. This returns the normal of the Sphere at the point intersectPoint. Uses cheap method of unitising sphere normals
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint	{	return [[intersectPoint subtractVector:centre] divideByFloat:radius];	}

//Debugging
- (void)logValues
{
	[super logValues];
	NSLog(@"Centre: (%1.3f, %1.3f, %1.3f), Radius: %f", [centre x], [centre y], [centre z], radius);
}

@end
