//
//  SphereMapping.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Mon Oct 15 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "SphereMapping.h"


@implementation SphereMapping

// Class methods
+ (SphereMapping *)sphereMappingWithDictionary:(NSDictionary *)dictionary
{	return [[[SphereMapping alloc] initWithDictionary:dictionary] autorelease];		}

- (id)copyWithZone:(NSZone *)zone	{	return [[SphereMapping allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

// Used when we are loading a scene file
- (id)initWithDictionary:(NSDictionary *)dictionary
{
	//initialize the object using the super class's method
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		// We use this so the mapped image does not need to be in a 2:1 ratio
		//float imageRatio;
		pixelsWide = [image pixelsWide];
		pixelsHigh = [image pixelsHigh];
		imageRatio = pixelsWide / pixelsHigh;

		UMult = VMult = 1;
	
		if (imageRatio > 2)		// Width is greater than 2 in 2:1 ratio
		{
			// This crops the image, but is also used in scaling
			pixelsWide *= 2 / imageRatio;

			// If we want to scale the image we have to change the U multiplier
			if (proportion == SCALED)
				UMult = imageRatio / 2;
		}
		else					// Height is greater than 1 in 2:1 ratio
		{
			// This crops the image, but is also used in scaling
			pixelsHigh *= imageRatio / 2;

			// If we want to scale the image we have to change the V multiplier
			if (proportion == SCALED)
				VMult = 2 / imageRatio;
		}

		return self;
	}
	return nil; // something went wrong.
}

// Used when we are saving to a scene file
- (NSMutableDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:@"Sphere Mapping" forKey:@"Material Type"];
	return dictionary;
}

// Convert the hitPoint to U and V coords of the image
- (NSPoint)getPixelCoords:(Vector3D *)hitPoint
{
	NSPoint pixelCoords;
	float U, V;
	float theta = (float)atan2([hitPoint y], [hitPoint x]);
	float phi = (float)acos([hitPoint z]);
	
	if (theta >= 0.0)
		U = theta / (2 * PI);
	else
		U = 1 + theta / (2 * PI);

	V = 1 - phi / PI;

	// Map the U and V to pixel coords
	pixelCoords.x = (pixelsWide - 1) * U * UMult;
	pixelCoords.y = (pixelsHigh - 1) * (1 - V) * VMult;

//	NSLog(@"Mapping U = %f, V = %f, pixelCoords.x = %f, .y = %f, to hitPoint:", U, V, pixelCoords.x, pixelCoords.y);
//	[hitPoint logValues];

	return pixelCoords;
}

- (RGBVector *)diffuseColour:(Vector3D *)hitPoint
{
	NSPoint coords = [self getPixelCoords:hitPoint];
	unsigned int byteOffset = ([image bitsPerPixel] / 8) * (((int)coords.y * [image pixelsWide]) + (int)coords.x);
	unsigned char *bitmap = [image bitmapData];
//	NSLog(@"Booty offset: %u, red: %f, green: %f, blue: %f", byteOffset, bitmap[byteOffset]/256.0, bitmap[byteOffset+1]/256.0, bitmap[byteOffset+2]/256.0);
	return ([RGBVector rgbVectorWithValuesRed:(bitmap[byteOffset]/255.0) green:(bitmap[byteOffset+1]/255.0) blue:(bitmap[byteOffset+2]/255.0)]);
}

@end
