//
//  Triangle.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Wed Sep 05 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Triangle.h"


@implementation Triangle
// Class methods
+ (Triangle *)triangleWithDictionary:(NSDictionary *)dictionary
{	return [[[Triangle alloc] initWithDictionary:dictionary] autorelease];	}

- (id)init
{
	//initialize the object using the super class's method
	if (self = [super init])
	{	// super init worked.
		// allocate memory for the verticies
		vertexA = [[Vector3D alloc] init];
		vertexB = [[Vector3D alloc] init];
		vertexC = [[Vector3D alloc] init];
		normal = [[Vector3D alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// init with dictionary takes a dictionary and initialises the data with this dictionary. This is the most common init method for us as it is really useful when parsing our scene files.
- (id)initWithDictionary:(NSDictionary *)dictionary	//initialize the object using the super class's method
{
	if (self = [super initWithDictionary:dictionary])
	{	// super init worked.
		vertexA = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Vertex A"]];
		vertexB = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Vertex B"]];
		vertexC = [[Vector3D alloc] initWithArray:[dictionary valueForKey:@"Vertex C"]];
		normal = [[Vector3D alloc] init];
		[self calculateNormal];
		return self;
	}
	return nil; // something went wrong.
}

// copyWithZone is needed for use by NSOutlineView which is used in our editor drawer. All classes which appear in the NSOutlineView will implement this method.
- (id)copyWithZone:(NSZone *)zone	{	return [[Triangle allocWithZone:zone] initWithDictionary:[self asDictionary]];	}

// Destructor
- (void)dealloc
{
	[vertexA release];
	[vertexB release];
	[vertexC release];
	[normal release];
	//use the super class's method for instance destruction
	[super dealloc];
}

// as dictionary returns all the data within this object as a dictionary. Really useful when saving a scene file.
- (NSDictionary *)asDictionary
{
	NSMutableDictionary *dictionary = [super asDictionary];
	[dictionary setObject:[vertexA asArray]	forKey:@"Vertex A"];
	[dictionary setObject:[vertexB asArray]	forKey:@"Vertex B"];
	[dictionary setObject:[vertexC asArray]	forKey:@"Vertex C"];
	[dictionary setObject:@"Triangle" forKey:@"Object Type"];
	[self calculateNormal];
	return dictionary;
}
// Assignment method
- (void)assign:(Triangle *)rhs
{
	[vertexA assign:[rhs vertexA]];
	[vertexB assign:[rhs vertexB]];
	[vertexC assign:[rhs vertexC]];
	[self calculateNormal];
}

// Setters. Everytime you move a point you want to calculate the new normal
- (void)setWithVectorsA:(Vector3D *)newA b:(Vector3D *)newB c:(Vector3D *)newC
{
	[vertexA assign:newA];
	[vertexB assign:newB];
	[vertexC assign:newC];
	[self calculateNormal];
}
- (void)setVertexA:(Vector3D *)newA		{	[vertexA assign:newA];	[self calculateNormal];		}
- (void)setVertexB:(Vector3D *)newB		{	[vertexB assign:newB];	[self calculateNormal];		}
- (void)setVertexC:(Vector3D *)newC		{	[vertexC assign:newC];	[self calculateNormal];		}

// Getters
- (Vector3D *)vertexA	{	return vertexA;		}
- (Vector3D *)vertexB	{	return vertexB;		}
- (Vector3D *)vertexC	{	return vertexC;		}

// The normal to a plane is the same no matter where you hit it, so we calculate and store it here and then return the stored value
- (void)calculateNormal
{	[normal assign:[[[vertexB subtractVector:vertexA] crossProduct:[vertexC subtractVector:vertexA]] unit]];	}

// Intersection routine for the triangle
- (BOOL)intersects:(Ray *)ray
{
	float t;	// Ray parameter
	float a = [vertexA x] - [vertexB x], b = [vertexA x] - [vertexC x], c = [[ray direction] x], d = [vertexA x] - [[ray origin] x];
	float e = [vertexA y] - [vertexB y], f = [vertexA y] - [vertexC y], g = [[ray direction] y], h = [vertexA y] - [[ray origin] y];
	float i = [vertexA z] - [vertexB z], j = [vertexA z] - [vertexC z], k = [[ray direction] z], l = [vertexA z] - [[ray origin] z];

	float m = f * k - g * j, n = h * k - g * l, p = f * l - h * j;
	float q = g * i - e * k, r = e * l - h * i, s = e * j - f * i;
	
	float e1 = d * m - b * n - c * p;
	float e2 = a * n + d * q + c * r;
	float D = a * m + b * q + c * s;

	float beta = e1 / D, gamma = e2 / D;

	if ((beta + gamma < 1.0) && (beta > 0.0) && (gamma > 0.0))
	{
		// This is your calculation of e3
		t = (a * p - b * r + d * s) / D;
		
		if (t > EPSILON)
		{
			[hitPoint assign:[[ray origin] addVector:[[ray direction] multiplyByFloat:t]]];
			[ray setT:t];
			return YES;
		}
	}
	return NO;
}

// I wouldn't have thought that is was possible, but this is even easier than for plane
- (Vector3D *)normalAtPoint:(Vector3D *)intersectPoint					{	return normal;	}

@end
