//
//  Vector3D.h
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Tue Aug 07 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "Transformation.h"

// This class is used for both points and directions which are mathematically quite similar. Also this cuts down on repeated code
@interface Vector3D : NSObject {
	// Basic components of a vector
	float x, y, z;
}

// Class methods
+ (Vector3D *)vector3DWithValuesX:(float)newX y:(float)newY z:(float)newZ;
+ (Vector3D *)vector3DWithArray:(NSArray *)array;
+ (Vector3D *)vector3DWithLinearFrameStart:(Vector3D *)start end:(Vector3D *)end currentFrame:(int)currentFrame totalFrames:(int)totalFrames;

// Initialisation with values for the components
- (id)initWithValuesX:(float)newX y:(float)newY z:(float)newZ;
// Initialisation with another vector
- (id)initWithVector:(Vector3D *)newVector;
- (id)initWithArray:(NSArray *)array;
// Assignment method
- (void)assign:(Vector3D *)rhs;
// Returns an array of NSNumbers
- (NSArray *)asArray;
// Setter with values
- (void)setWithValuesX:(float)newX y:(float)newY z:(float)newZ;
// Set individual components
- (void)setX:(float)newX;
- (void)setY:(float)newY;
- (void)setZ:(float)newZ;
// Get individual components
- (float)x;
- (float)y;
- (float)z;
// Normalise the current vector
- (void)normalise;
// Used mainly for directional vectors, returns the vector of unit length
- (Vector3D *)unit;
// The length of the vector
- (float)length;
// Mathematical constructs for vector manipulation
- (Vector3D *)addVector:(Vector3D *)rhs;
- (Vector3D *)subtractVector:(Vector3D *)rhs;
- (Vector3D *)multiplyByFloat:(float)rhs;
- (Vector3D *)divideByFloat:(float)rhs;
- (Vector3D *)divideByVector:(Vector3D *)rhs;

- (float)dotProduct:(Vector3D *)rhs;
- (Vector3D *)crossProduct:(Vector3D *)rhs;

// Transformation functions
- (Vector3D *)transformAsPoint:(Transformation *)transformation;
- (Vector3D *)transformAsVector:(Transformation *)transformation;
- (Vector3D *)transformAsNormal:(Transformation *)transformation;

// Debugging
- (void)logValues;

@end
