//
//  Vector3D.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Tue Aug 07 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "Vector3D.h"


@implementation Vector3D
// Class methods
+ (Vector3D *)vector3DWithValuesX:(float)newX y:(float)newY z:(float)newZ
{	return [[[self alloc] initWithValuesX:newX y:newY z:newZ] autorelease];		}

+ (Vector3D *)vector3DWithArray:(NSArray *)array
{	return [[[self alloc] initWithArray:array] autorelease];	}

+ (Vector3D *)vector3DWithLinearFrameStart:(Vector3D *)start end:(Vector3D *)end currentFrame:(int)currentFrame totalFrames:(int)totalFrames
{
	return [Vector3D vector3DWithValuesX:[start x] + (([end x] - [start x])/(totalFrames - 1))*currentFrame
									y:[start y] + (([end y] - [start y])/(totalFrames - 1))*currentFrame
									z:[start z] + (([end z] - [start z])/(totalFrames - 1))*currentFrame];
}

// Initialisation with values for the components
- (id)initWithValuesX:(float)newX y:(float)newY z:(float)newZ
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		x = newX;
		y = newY;
		z = newZ;
		return self;
	}
	return nil; // something went wrong.
}

// Initialisation with another vector
- (id)initWithVector:(Vector3D *)newVector;
{
	if (self = [super init])	//initialize the object using the super class's method
	{	// super init worked.
		x = [newVector x];
		y = [newVector y];
		z = [newVector z];
		return self;
	}
	return nil; // something went wrong.
}

// Super init is called within initWithValues
- (id)initWithArray:(NSArray *)array
{
	return [self initWithValuesX:[[array objectAtIndex:0] floatValue]
			y:[[array objectAtIndex:1] floatValue]
			z:[[array objectAtIndex:2] floatValue]];
}

// Assignment method
- (void)assign:(Vector3D *)rhs
{
	x = [rhs x];
	y = [rhs y];
	z = [rhs z];
}

- (NSArray *)asArray
{	return [NSArray arrayWithObjects:[NSNumber numberWithFloat:x], [NSNumber numberWithFloat:y], [NSNumber numberWithFloat:z], nil];	}

// *** Setter methods ***
- (void)setWithValuesX:(float)newX y:(float)newY z:(float)newZ
{
	x = newX;
	y = newY;
	z = newZ;
}

- (void)setX:(float)newX	{	x = newX;	}
- (void)setY:(float)newY	{	y = newY;	}
- (void)setZ:(float)newZ	{	z = newZ;	}

// *** Getter methods ***
- (float)x					{	return x;	}
- (float)y					{	return y;	}
- (float)z					{	return z;	}

// Normalise the current vector
- (void)normalise
{
	float length = [self length];
	x /= length;
	y /= length;
	z /= length;
}

// Returns the unit vector of the current object (ie divided by its own length)
- (Vector3D *)unit	{	return [self divideByFloat:[self length]];	}

// Returns the length
- (float)length		{	return sqrt(x * x + y * y + z * z);			}

// Adds another vector to the vector
- (Vector3D *)addVector:(Vector3D *)rhs
{	return [Vector3D vector3DWithValuesX:(x + [rhs x]) y:(y + [rhs y]) z:(z + [rhs z])];	}

// Subtract another vector from the vector
- (Vector3D *)subtractVector:(Vector3D *)rhs
{	return [Vector3D vector3DWithValuesX:(x - [rhs x]) y:(y - [rhs y]) z:(z - [rhs z])];	}

// Multiply the vector by a float
- (Vector3D *)multiplyByFloat:(float)rhs
{	return [Vector3D vector3DWithValuesX:(x * rhs) y:(y * rhs) z:(z * rhs)];				}

// Divides the vector by a float
- (Vector3D *)divideByFloat:(float)rhs
{	return [Vector3D vector3DWithValuesX:(x / rhs) y:(y / rhs) z:(z / rhs)];				}

// Divides the vector by another Vector
- (Vector3D *)divideByVector:(Vector3D *)rhs
{	return [Vector3D vector3DWithValuesX:(x / [rhs x]) y:(y / [rhs y]) z:(z / [rhs z])];	}

// Dot product of two vectors
- (float)dotProduct:(Vector3D *)rhs		{	return (x * [rhs x] + y * [rhs y] + z * [rhs z]);	}

// Cross product of two vectors
- (Vector3D *)crossProduct:(Vector3D *)rhs
{	return [Vector3D vector3DWithValuesX:(y * [rhs z] - z * [rhs y]) y:(z * [rhs x] - x * [rhs z]) z:(x * [rhs y] - y * [rhs x])];	}

// Transformation functions
- (Vector3D *)transformAsPoint:(Transformation *)transformation
{
	return [Vector3D vector3DWithValuesX:
		(x * [transformation valueAtRow:0 col:0] + y * [transformation valueAtRow:1 col:0] + z * [transformation valueAtRow:2 col:0] + [transformation valueAtRow:3 col:0])
		y:(x * [transformation valueAtRow:0 col:1] + y * [transformation valueAtRow:1 col:1] + z * [transformation valueAtRow:2 col:1] + [transformation valueAtRow:3 col:1])
		z:(x * [transformation valueAtRow:0 col:2] + y * [transformation valueAtRow:1 col:2] + z * [transformation valueAtRow:2 col:2] + [transformation valueAtRow:3 col:2])
		];
}

- (Vector3D *)transformAsVector:(Transformation *)transformation
{
	return [Vector3D vector3DWithValuesX:
		(x * [transformation valueAtRow:0 col:0] + y * [transformation valueAtRow:1 col:0] + z * [transformation valueAtRow:2 col:0])
		y:(x * [transformation valueAtRow:0 col:1] + y * [transformation valueAtRow:1 col:1] + z * [transformation valueAtRow:2 col:1])
		z:(x * [transformation valueAtRow:0 col:2] + y * [transformation valueAtRow:1 col:2] + z * [transformation valueAtRow:2 col:2])
		];
}

- (Vector3D *)transformAsNormal:(Transformation *)transformation
{
	return [Vector3D vector3DWithValuesX:
		(x * [transformation valueAtRow:0 col:0] + y * [transformation valueAtRow:0 col:1] + z * [transformation valueAtRow:0 col:2])
		y:(x * [transformation valueAtRow:1 col:0] + y * [transformation valueAtRow:1 col:1] + z * [transformation valueAtRow:1 col:2])
		z:(x * [transformation valueAtRow:2 col:0] + y * [transformation valueAtRow:2 col:1] + z * [transformation valueAtRow:2 col:2])
		];
}

// Logs the vector Values
- (void)logValues	{	NSLog(@"X = %f, Y = %f, Z = %f", x, y, z);	}
@end
