//
//  World.h
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import "All3DObjects.h"
#import "Ray.h"
#import "RGBVector.h"
#import "Preferences.h"
#import "Vector3D.h"
#import "Camera.h"
#import "Light.h"
#import "SceneParser.h"
#import "ShadingObject.h"
#import "RayMovie.h"
#import <QuickTime/QuickTime.h>
#import "QuickTimeCode.h"


#define SAMPLE_SIZE 8
#define SAMPLE_NUMBER 3
#define PIXEL_BYTES 3

#define MAX_DISTANCE 1000000
#define TOLERANCE 0.05

#define PREVIEW 0
#define SIGNATURE 1
#define RENDER 2

#define PIXEL_SIZE 1


@interface World : NSObject
{
	//Used for storing our Materials. Note: this was done as association with the objects rather than composition so that multiple objects could use one material
	//Used for storing our objects, lights
	//scene returns camera, objects and lights as an array for use by the controller
	//Image array is used for creating our movie but as yet is still a work in progress
	NSMutableArray *threeDObjects, *lights, *scene, *imageArray, *materials;
	//stores the current user preferences in memory
	Preferences *loadedPreferences;
	//Preview, Signature, or Render the scene. Signature is not implemented yet (although it would not be hard to).
	int renderStyle;
	//Stores the camera location
	Camera *camera;
	// The colour of the ambient light. We didn't see the need to create an entire class just for this/ nor background colour
	RGBVector *ambientLight, *backgroundColour;
	//This is a pointer to the bitmapData that the controller has allocated in memory. The controller is responsible for doing this as it needs to realse and reallocate the memory if the user changes the size of the image in the preferences. By giving World a pointer, we can directly write to this memory.
	unsigned char *bitmapData;
	
	// These are used in the calculation of a ray from the camera, we don't need to calculate these each time
	float sinTheta, cosTheta, sinPhi, cosPhi, r, twist, d, sinTwist, cosTwist;
	//Used for analysis of times taken when rendering
	double totRef, totTra, totIntersects, totCam;
}

// Initialise the world passing through the use preferences from the controller. The controller loads the preferences as these are needed before any scene is loaded or rendered.
- (id)initWithPreferences:(Preferences *)loadedPrefs;
// Getter for the materials array
- (NSArray *)materials;
// Getter for bitmapData (this should not be needed as the controller already has this pointer but we put it in for OO reasons?)
- (unsigned char *)bitmapData;
// Setter for the pointer to bitmapData
- (void)setBitmapData:(unsigned char *)newBitmapData;
// Build will create a sceneparser object and then use it to populate our scene arrays
- (BOOL)BuildWithName:(NSString *)filename;
- (BOOL)buildMaterials:(NSString *)filename;
// Demolish is the opposite of build as it will remove all objects from our scene. Needed when closing a scene file in order to open another scene.
- (void)Demolish;
// Saves the current scene to an XML file via the Scene Parser class.
- (BOOL)saveSceneWithName:(NSString *)filename;
// Generate Scene is an intermediatory step to decide whether to Render or Signature... (not really needed yet).
- (void)GenerateScene:(int)newRenderStyle;
// This method will set the colour of the pixel in our bitmap data. It knows exactly how the pixels are stored in the data (sample size, pixel size etc)
- (void)setPixelWithValuesColour:(RGBVector *)colour row:(int)row col:(int)col blockSize:(int)blockSize;
// As yet is not implemented but will simply do a first hit ray trace without shading or lights etc
- (void)SignatureScene;
// This is where most of the work is done. It will render the whole scene and notify our controller to draw it.
- (void)RenderScene;
// Calculate the ray that we wish to trace. Row and col are floats so we can do anti-aliasing which relies on subpixel size
- (Ray *)cameraRayWithRow:(float)row col:(float)col;
// A work in progress. At the moment it takes a start and end frame and calculates a linear tween between the two scene files. Currently camera only 
- (void)renderMovieWithNumberOfKeyFrames:(int)numKeyFrames numberOfFrames:(int)numFrames size:(NSRect) viewRect;
// These two methods will be used by the Quicktime API when we get it working. They are accessor methods for our image array
- (id)getImageAtArrayIndex:(int)index;
- (int)getNumberOfImages;
// Used when the a user clicks on the image to find a specific object
- (Object3D *)findObjectAtRow:(int)row col:(int)col;
// Our recursive function that will return the colour of each pixel at the intersect of each ray
- (RGBVector *)TraceRay:(Ray *)ray depth:(int)depth;
// Returns true if the ray intersects an object. It is also passed a shading object pointer. This is so the information about the intersection point can be stored in the shading object. We say passing a pointer to the shading object a better method than returning a shading object.
- (BOOL)IntersectObjects:(Ray *)ray shader:(ShadingObject *)shader;
// Getter for the scene
- (NSMutableArray *)scene;
//Calculates the reflected ray direction of the incident ray and the intersection
- (Vector3D *)reflectedRayDirection:(Ray *)incidentRay shader:(ShadingObject *)shader;
// Calculates the transmitted ray direction
- (Vector3D *)transmittedRayDirection:(Ray *)incidentRay shader:(ShadingObject *)shader;

//Adds objects to the scene.
- (void)addSphere;
- (void)addPlane;
- (void)addLight;
- (void)addTriangle;
- (void)addGenericBox;
- (void)addGenericSphere;
- (void)addGenericTorus;
- (void)addGenericTaperedCylinder;
@end
