//
//  World.m
//  RayTracer
//
//  Created by Stuart Bryson and Tim Keighley on Fri Aug 03 2001.
//  Copyright (c) 2001 Stuart Bryson and Tim Keighley. All rights reserved.
//

#import "World.h"


@implementation World

// Initialise the world passing through the use preferences from the controller. The controller loads the preferences as these are needed before any scene is loaded or rendered.
- (id)initWithPreferences:(Preferences *)loadedPrefs
{
	if (self = [super init])	//initialize the object using the super class's method
	{
		loadedPreferences = loadedPrefs; // set prefs pointer
		camera = [[Camera alloc] init];
		materials = [[NSMutableArray alloc] init];	//init materials dictionary
		threeDObjects = [[NSMutableArray alloc] init];	//init object array 
		lights = [[NSMutableArray alloc] init];			//init lights array
		imageArray = [[NSMutableArray alloc] init];		//not being used atm, will be used for Quicktime API (at some stage)
		ambientLight = [[RGBVector alloc] init];
		backgroundColour = [[RGBVector alloc] init];
		return self;
	}
	return nil; // something went wrong.
}

// Destructor
- (void)dealloc
{
	[camera release];
	//release threeDObjects array, dealloc is called when the object is going to be destroyed
	[threeDObjects release];
	[lights release];
	[imageArray release];
	// We don't need to release the objects in the array as they were inserted into the array with "retain" and the array will release them when the array is released.
	[ambientLight release];
	[backgroundColour release];
	[super dealloc];  //use the super class's method for instance destruction
}

// Getter for the materials array
- (NSArray *)materials	{	return materials;	}

// Getter for bitmapData (this should not be needed as the controller already has this pointer but we put it in for OO reasons?)
- (unsigned char *)bitmapData
{	return bitmapData;	}

// Setter for the pointer to bitmapData
- (void)setBitmapData:(unsigned char *)newBitmapData
{	bitmapData = newBitmapData;	}

- (NSMutableArray *)scene
{
	//***** NOTE - REORDERING THESE WILL REQUIRE CHANGES TO DELETE OBJECT IN THE CONTROLLER
	return [[NSMutableArray arrayWithObjects:threeDObjects, lights, materials, camera, nil] retain];
}

// Build will create a sceneparser object and then use it to populate our scene arrays
// Returns true if the scene file was successfully imported
- (BOOL)BuildWithName:(NSString *)filename
{
    Material *defaultMaterial = [[Material alloc] init];
    [defaultMaterial setName:@"Default Material"];
    [materials addObject:defaultMaterial];
    [defaultMaterial release];
    if (filename != nil)
    {
        SceneParser *newScene = [[SceneParser alloc] init];
        if ([newScene importSceneWithCamera:camera materials:materials objects:threeDObjects
            lights:lights ambient:ambientLight background:backgroundColour filename:filename])
        {
            [newScene release];
            return YES;
        }
        [newScene release];
    }
    else
    {
        [materials addObject:@"Materials"];
        [lights addObject:@"Lights"];
        [threeDObjects addObject:@"3D Objects"];
        return YES;
    }
    return NO;
}

- (BOOL)buildMaterials:(NSString *)filename
{
    if (filename != nil)
    {
        SceneParser *newScene = [[SceneParser alloc] init];
        if ([newScene importMaterialLibrary:materials filename:filename])
        {
            [newScene release];
            return YES;
        }
        [newScene release];
    }
    return NO;
}

// A work in progress. At the moment it takes a start and end frame and calculates a linear tween between the two scene files. Currently camera only 
- (void)renderMovieWithNumberOfKeyFrames:(int)numKeyFrames numberOfFrames:(int)numFrames size:(NSRect)viewRect
{
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
	RayMovie *rayMovie = [[RayMovie alloc] initWithNumberOfKeyFrames:numKeyFrames numberOfFrames:numFrames];
	int i;
	NSDictionary *frame;
	//NSMutableData *movieImageBitmap=[NSMutableData dataWithLength:sizeof(bitmapData)];
	renderStyle = RENDER;
	
	for (i = 0; i < numFrames; i++)
	{
		// Get the next frame in the tween
		frame = [rayMovie getNextFrame];
		[camera assign:[frame valueForKey:@"Camera"]];
		[threeDObjects setArray:[frame valueForKey:@"Objects"]];
		[lights setArray:[frame valueForKey:@"Lights"]];
		[ambientLight assign:[frame valueForKey:@"Ambient Light"]];
		[backgroundColour assign:[frame valueForKey:@"Background Colour"]];
		// Render that frame
		[self RenderScene];
		// Eventually we will add this new image to our array and pass it to our quicktime method to generate a movie.
		//[imageArray addObject:movieImageBitmap];
	//This is just temporary until we get the quicktime api working, we will just write each frame to a file.
	{	
		NSBitmapImageRep *theBitmapRep = [NSBitmapImageRep alloc];
		NSImage *theImage;
		NSData *tiffImage;
		[theBitmapRep initWithBitmapDataPlanes:&bitmapData
			pixelsWide:[loadedPreferences pixelsWide]
			pixelsHigh:[loadedPreferences pixelsHigh] 
			bitsPerSample:SAMPLE_SIZE      
			samplesPerPixel:SAMPLE_NUMBER
			hasAlpha:NO
			isPlanar:NO
			colorSpaceName:NSCalibratedRGBColorSpace
			bytesPerRow:([loadedPreferences pixelsWide] * PIXEL_BYTES)
			bitsPerPixel:0];
		//This line does not work yet... thus we end up with tiffs that are of length 0k. We will fix it soon but not urgent.
		theImage = [[NSImage alloc] initWithSize:viewRect.size];
		[theImage addRepresentation:theBitmapRep];
		tiffImage = [[[NSData alloc] initWithData:[theImage TIFFRepresentation]] autorelease];
		// Just save the file as MovieFrameX where x is the iterator
		[tiffImage writeToFile:[[[NSString stringWithString:@"/Users/spbryson/MovieFrame"] stringByAppendingString:[[NSNumber numberWithInt:i] stringValue]] stringByAppendingString:@".tiff"] atomically:YES];
		
		[theImage release];
		[theBitmapRep release];
	
	}	
		//Tell our controller to draw the new image. This works differently to the render call made by the controller as it only needs to get one picture returned however here we need to draw multiple pictures to the screen so instead we use a notification.
		[nc postNotificationName: @"drawNotification" object:nil];
	//add to movie
	}
	[rayMovie release];
}

// These two methods will be used by the Quicktime API when we get it working. They are accessor methods for our image array
- (id)getImageAtArrayIndex:(int)index	{	return [imageArray objectAtIndex:index];	}
- (int)getNumberOfImages				{	return [imageArray count];					}

// Demolish is the opposite of build as it will remove all objects from our scene. Needed when closing a scene file in order to open another scene.
- (void)Demolish
{
	[threeDObjects removeAllObjects];
	[materials removeAllObjects];
	[lights removeAllObjects];
	return;
}

// Saves the current scene to an XML file via the Scene Parser class.
// Currently this should only be called when closing a scene. ie. You can't save a scene unless you close it as well... not the best but we will fix it soon.
- (BOOL)saveSceneWithName:(NSString *)filename
{
	SceneParser *newScene = [[SceneParser alloc] init];
	if ([newScene saveSceneWithCamera:camera materials:materials objects:threeDObjects
			lights:lights ambient:ambientLight background:backgroundColour filename:filename])
		[newScene release];
	else
		return NO;
	return YES;
}

// Generate Scene is an intermediatory step to decide whether to Render or Signature... (not really needed yet).
- (void)GenerateScene:(int)newRenderStyle
{
	renderStyle = newRenderStyle;
	if (renderStyle == PREVIEW || renderStyle == RENDER)
		[self RenderScene];
	else if (renderStyle == SIGNATURE)
		[self SignatureScene];
}

// This does most of the hard work. The method could be a lot more compact however we are encapsulating a lot of functionality that would not be worth splitting up into multiple functions. This is the main loop that controls the tracing of rays, antialiasing and previewing our scene. We will also look at adding in threads here as this will speed up the render time on a multiprocessor machine (as mine is - Stuart).
- (void)RenderScene
{
	// Declare some local variables to make execution faster and code easier to read
	int row, col, previewBlockSize, pixelsHigh, pixelsWide;
	RGBVector *colour = [[RGBVector alloc] init];
	Ray *ray = [[Ray alloc] init];
	NSMutableArray *pixels = [[NSMutableArray alloc] init];
	
	NSDate *startRender = [NSDate date];
	// This autorelease pool is to do with memory management. When using autoreleased objects such as those returned from Vector3D when we call dotProduct for example, these objects are not released from memory until the end of the main loop (ie. after every ray is shot). This is far from optimal. For example we had a 320*240 image that instantiated 3 million Vector3D objects in memory at one time. After implementing our own autorelease pool and specifying to release after each ray, we only had 87 Vector3D objects in memory at one time.
	NSAutoreleasePool *pool;
	
	//Set up our notification centre
	NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
	
	// If the render is a full render set the previewBlockSize to one pixel, else set it to the size specified in the preferences
	if (renderStyle == RENDER)
		previewBlockSize = 1;
	else
		previewBlockSize = [loadedPreferences previewBlockSize];
	
	pixelsWide = [loadedPreferences pixelsWide];
	pixelsHigh = [loadedPreferences pixelsHigh];
	
	if (![loadedPreferences useParallelProjection])	// Use Parallel projection
	{
		float camAE = [camera a] - [camera e], camBF = [camera b] - [camera f], camCG = [camera c] - [camera g];
		float denom = sqrt(camAE * camAE + camBF * camBF);

		r = sqrt(camAE * camAE + camBF * camBF + camCG * camCG);
		sinTheta = camBF / denom;
		cosTheta = camAE / denom;
		sinPhi = denom / r;
		cosPhi = camCG / r;
		twist = [camera twist];
		sinTwist = sin(twist);
		cosTwist = cos(twist);
		d = [camera d];
	}
	
	for (row = 0; row <= (pixelsHigh - 1); row += previewBlockSize)	// for every row
	{
		//This line notifies our controller to update the progress bar?? If we use it and also to redraw the image with the newly rendered row
		[nc postNotificationName: @"updateProgress" object:[NSDictionary dictionaryWithObject:(id)[NSNumber numberWithInt:(row)] forKey:(id)@"pixels"]];
		for (col = 0; col <= (pixelsWide - 1); col += previewBlockSize) // for every col
		{
			pool = [[NSAutoreleasePool alloc] init];	//init our autorelease pool (as discussed above)
			[ray assign:[self cameraRayWithRow:row col:col]];
			//Add colours to an NSArray so that we can AntiAlias them easily with the second pass
			[pixels addObject:[self TraceRay:ray depth:[loadedPreferences recursionDepth]]];
			[self setPixelWithValuesColour:[pixels objectAtIndex:([pixels count]-1)] row:row col:col blockSize:previewBlockSize];
			// Release the autorelease objects that are in memory
			[pool release];
		}	// end loop for columns
	}		// end loop for rows

	// Adaptive Antialiasing - Super Sampling
	// We only want to do this if we are doing a full render
	if (renderStyle == RENDER && [loadedPreferences antiAlias])
	{
		RGBVector *temp = [[RGBVector alloc] init];
		// Testing row against row + 1
		for (row = 0; row < (pixelsHigh - 1); row++)	// for every row
		{
			//This line notifies our controller to update the progress bar?? If we use it and also to redraw the image with the newly rendered row
			[nc postNotificationName: @"updateProgress" object:[NSDictionary dictionaryWithObject:(id)[NSNumber numberWithInt:(row)] forKey:(id)@"pixels"]];
			// Testing col against col + 1
			for (col = 0; col < (pixelsWide - 1); col++) // for every col
			{
				// The corners are numbered according to foil 6 of "Adaptive Super-Sampling"
				//	D	-	E
				//	|		|
				//	A	-	B

				// Test D - E
				if ([[pixels objectAtIndex:(col + row * pixelsWide)] toleranceCompare:[pixels objectAtIndex:((col+1) + (row+0) * pixelsWide)] tolerance:TOLERANCE] ||
				// Test D - A
				[[pixels objectAtIndex:(col + row * pixelsWide)] toleranceCompare:[pixels objectAtIndex:((col+0) + (row+1) * pixelsWide)] tolerance:TOLERANCE] ||
				// Test D - B
				[[pixels objectAtIndex:(col + row * pixelsWide)] toleranceCompare:[pixels objectAtIndex:((col+1) + (row+1) * pixelsWide)] tolerance:TOLERANCE])
				{
					pool = [[NSAutoreleasePool alloc] init];	//init our autorelease pool (as discussed above)
//					NSLog(@"Alias at row: %i col: %i", row, col);
//					this pixel = TraceRay blah + traceRay bvlah + traceRay blah + trace bal
					// D
					[temp assign:[pixels objectAtIndex:(col + row * pixelsWide)]];
					// E
					[temp assign:[temp averageColour:[pixels objectAtIndex:((col+1) + row * pixelsWide)]]];
					// A
					[temp assign:[temp averageColour:[pixels objectAtIndex:(col + (row+1) * pixelsWide)]]];
					// B
					[temp assign:[temp averageColour:[pixels objectAtIndex:((col+1) + (row+1) * pixelsWide)]]];

					// Between D and E
					[temp assign:[temp averageColour:[self TraceRay:[self cameraRayWithRow:row col:(col+0.5)] depth:[loadedPreferences recursionDepth]]]];
					// Between D and A
					[temp assign:[temp averageColour:[self TraceRay:[self cameraRayWithRow:(row+0.5) col:col] depth:[loadedPreferences recursionDepth]]]];
					// Between A and B
					[temp assign:[temp averageColour:[self TraceRay:[self cameraRayWithRow:(row+1) col:(col+0.5)] depth:[loadedPreferences recursionDepth]]]];
					// Between E and B
					[temp assign:[temp averageColour:[self TraceRay:[self cameraRayWithRow:(row+0.5) col:(col+1)] depth:[loadedPreferences recursionDepth]]]];
					// Between D and B
					[temp assign:[temp averageColour:[self TraceRay:[self cameraRayWithRow:(row+0.5) col:(col+0.5)] depth:[loadedPreferences recursionDepth]]]];

					// Now we actually set the pixel with the new value
					[self setPixelWithValuesColour:temp row:row col:col blockSize:previewBlockSize];

					// Release the autorelease objects that are in memory
					[pool release];
				}	// End if pixels are out of tolerance
			}		// End for cols
		}			// End for rows
		[temp release];
	}				// End if renderStyle == RENDER
	[pixels release];
	[colour release];
	[ray release];
	
	
	//NSLog(@"Total Render Time: %f",(-(double)[startRender timeIntervalSinceNow]));
	return;
}

// This method will set the colour of the pixel in our bitmap data. It knows exactly how the pixels are stored in the data (sample size, pixel size etc)
- (void)setPixelWithValuesColour:(RGBVector *)colour row:(int)row col:(int)col blockSize:(int)blockSize
{
	unsigned int i, byteOffset, intColor[3];
	int rowBlock, colBlock, pixelsWide;
	float color[3];
	double intPart;

	pixelsWide = [loadedPreferences pixelsWide];

	//Put RGBVector into an array of floats (easier to manipulate at this level)
	color[0] = [colour red];
	color[1] = [colour green];
	color[2] = [colour blue];
	
	for (i=0; i < 3; i++)
	{
		//Do some final clamping
		if (color[i] > 1.0)
			color[i] = 1.0; 
		if (color[i] < 0.0)
			color[i] = 0.0;
		//Convert to 255 colours... this could be easily extended to thousands or millions
		color[i] *= 255; //Convert to unsigned int byte
		//Round the result to the nearest int and store it in a new array
		if (modf(color[i],&intPart) >= 0.5)
			intColor[i] = intPart + 1;
		else
			intColor[i] = intPart;
	}
	//If one ray is to colour more than one pixel, then we need to colour all the pixels. This is done with row and col block. We use this for the purposes of previewing our scene.
	for (rowBlock = 0; rowBlock <= (blockSize - 1); rowBlock++)
	{
		for (colBlock = 0; colBlock <= (blockSize - 1); colBlock++)
		{	
			// Determine the offset of the current pixel being written
			byteOffset = PIXEL_BYTES * (((row + rowBlock) * pixelsWide) + col + colBlock);
			//Write each RGB value next to each other in the file
			for (i = 0; i < 3; i++)
				bitmapData[byteOffset + i] = intColor[i];
		}
	}
}

// As yet is not implemented but will simply do a first hit ray trace without shading or lights etc
- (void)SignatureScene	{	NSLog(@"SignatureScene called");	}

// Calculate the ray that we wish to trace. Row and col are floats so we can do anti-aliasing which relies on subpixel size
- (Ray *)cameraRayWithRow:(float)row col:(float)col
{
	NSDate *startCam;
	int previewBlockSize, pixelsHigh, pixelsWide;
	float xp, yp;//, sinTheta, cosTheta, sinPhi, cosPhi, r, twist, d, sinTwist, cosTwist;
	Vector3D *dw = [[Vector3D alloc] init];
	Ray *ray = [[[Ray alloc] init] autorelease];
	
	startCam = [NSDate date];
	// If the render is a full render set the previewBlockSize to one pixel, else set it to the size specified in the preferences
	if (renderStyle == RENDER)
		previewBlockSize = 1;
	else
		previewBlockSize = [loadedPreferences previewBlockSize];
	
	pixelsWide = [loadedPreferences pixelsWide];
	pixelsHigh = [loadedPreferences pixelsHigh];
	
	if ([loadedPreferences useParallelProjection])	// Use Parallel projection
	{
		[ray setOriginWithValuesX:100.0 y:(float)(col - (pixelsWide / 2.0)) z:(float)((pixelsHigh / 2.0) - row)];
		[ray setDirectionWithVector:[[[camera lookAt] subtractVector:[camera location]] unit]];
	}
	else	// Use perspective projection
	{
		[ray setOriginWithVector:[camera location]];
		xp = PIXEL_SIZE * (-pixelsWide / 2 + col);	// calculate xp
		yp = PIXEL_SIZE * ( pixelsHigh / 2 - row - 1);	//calculate yp
		// Calculate the viewing coordinates
		[dw setX:(
				(-cosTwist*sinTheta - sinTwist*cosTheta*cosPhi) * xp +
				( sinTwist*sinTheta - cosTwist*cosTheta*cosPhi) * yp +
				(-cosTheta*sinPhi) * d
				)];
		[dw setY:(
				( cosTwist*cosTheta - sinTwist*sinTheta*cosPhi) * xp +
				(-sinTwist*cosTheta - cosTwist*sinTheta*cosPhi) * yp +
				(-sinTheta*sinPhi) * d
				)];
		[dw setZ:(
				( sinTwist*sinPhi) * xp +
				( cosTwist*sinPhi) * yp +
				(-cosPhi) * d
				)];
		[ray setDirectionWithVector:[dw unit]];
	}	// end else
	[dw release];
	
	totCam += (-(double)[startCam timeIntervalSinceNow]);
	return ray;
}

// Used when the a user clicks on the image to find a specific object
- (Object3D *)findObjectAtRow:(int)row col:(int)col
{
	float camAE = [camera a] - [camera e], camBF = [camera b] - [camera f], camCG = [camera c] - [camera g];
	float denom = sqrt(camAE * camAE + camBF * camBF);
	
	Ray *ray = [[Ray alloc] init];
	ShadingObject *shader = [[ShadingObject alloc] init];
	Object3D *theObject = nil;
	
	r = sqrt(camAE * camAE + camBF * camBF + camCG * camCG);
	sinTheta = camBF / denom;
	cosTheta = camAE / denom;
	sinPhi = denom / r;
	cosPhi = camCG / r;
	twist = [camera twist];
	sinTwist = sin(twist);
	cosTwist = cos(twist);
	d = [camera d];
	
	[ray assign:[self cameraRayWithRow:row col:col]];
	
	if ([self IntersectObjects:ray shader:shader])
	{
		theObject = [shader object];
	}
	
	[ray release];
	[shader release];
	return theObject;
}

// Our recursive function that will return the colour of each pixel at the intersect of each ray
- (RGBVector *)TraceRay:(Ray *)ray depth:(int)depth
{
	NSDate *startIntersects;
	NSDate *startRef;
	NSDate *startTra;
	//Init the colour to the background colour... at the moment it is just hard coded as black but we will work it into the scene description file soon.
	RGBVector *rayColour = [[RGBVector alloc] initWithValuesRed:0.0 green:0.0 blue:0.0];
	
	//If the depth is less than 1 then we do not need to trace another ray and we just return
	if (depth > 0)
	{
		// Set up some temporary variables
		RGBVector *localColour = [[RGBVector alloc] init];
		RGBVector *reflectedColour = [[RGBVector alloc] initWithValue:0.0], *transmittedColour = [[RGBVector alloc] initWithValue:0.0];
		Ray *reflectedRay = [[Ray alloc] init], *transmittedRay = [[Ray alloc] init];
		float Kr, Kt;
		//Set up our shading object
		//Should we pass the world here instead? We hit circular referencing in our imports so we couldn't.
		ShadingObject *shader = [[ShadingObject alloc] init];
		[shader setLights:lights];
		[shader setObjects:threeDObjects];
		[shader setCameraLocation:[camera location]];
		[shader setAmbientLight:ambientLight];

		// Intersect the ray with every object in the scene
		// If we do not intersect, we return the background colour... at this stage hard coded as black
		startIntersects = [NSDate date];
		if (![self IntersectObjects:ray shader:shader])
		{
        	[rayColour assign:backgroundColour];
		}
		else
		{
                        
			//Otherwise we assign the local colour to the shading object, if the intersect point is reflective then trace another ray.
			[localColour assign:[shader localColour]];

			startRef = [NSDate date];
			if ([[[shader object] material] isReflective:[shader transIntersection]])
			{
				// Set the direction and origin of the new ray. Call reflectedRayDirection to find the new direction
				[reflectedRay setDirectionWithVector:[self reflectedRayDirection:ray shader:shader]];
				[reflectedRay setOriginWithVector:[shader intersection]];
				// Assign reflected colour with the reflected ray traced
				[reflectedColour assign:[self TraceRay:reflectedRay depth:(depth-1)]];
			}
			
			startTra = [NSDate date];
			if ([[[shader object] material] isTransparent:[shader transIntersection]])
			{
				// Set the direction and origin of the new ray. Call transmittedRayDirection to find the new direction
				[transmittedRay setDirectionWithVector:[self transmittedRayDirection:ray shader:shader]];
				if (![shader killed])
				{
					[transmittedRay setOriginWithVector:[shader intersection]];
					[transmittedColour assign:[self TraceRay:transmittedRay depth:(depth-1)]];
					if ([shader inside])
						[transmittedColour assign:[transmittedColour multiplyByColour:[shader attenuationColour]]];
				}
			}
			
			// The global reflection of the object determines how much of the world will be seen in it through reflection
			if ([shader killed])
			{
				Kr = 1;
				Kt = 0;
			}
			else
			{
				Kr = [[[shader object] material] globalReflection:[shader transIntersection]];
				Kt = [[[shader object] material] globalTransmission:[shader transIntersection]];
			}
			[rayColour assign:[localColour addColour:[reflectedColour multiplyByFloat:Kr]]];
			[rayColour assign:[rayColour addColour:[transmittedColour multiplyByFloat:Kt]]];
		}
		//Release our objects
		[shader release];
		[localColour release];
		[reflectedColour release];
		[reflectedRay release];
		[transmittedRay release];
		[transmittedColour release];
	}
	//Return the calculated ray
	return [rayColour autorelease];
}

// Returns true if the ray intersects an object. It is also passed a shading object pointer. This is so the information about the intersection point can be stored in the shading object. We say passing a pointer to the shading object a better method than returning a shading object.
- (BOOL)IntersectObjects:(Ray *)ray shader:(ShadingObject *)shader
{
	float nearestDistance = MAX_DISTANCE;	//Defined above as a really big number
	BOOL intersected = NO;	//Assume no intersects to begin with
	NSEnumerator *en;	//array enumerator
	id obj;
	//int i;
	//For every object in our threeDObjects array
	for (en=[threeDObjects objectEnumerator]; obj=[en nextObject];)
	//for (i = 0; i < [threeDObjects count]-1; i++)
	{
		//obj = [threeDObjects objectAtIndex:i];
		// We need to test that the object within the array is of type Object3D (or subclass). This is because in order to implement NSOutlineView (which lists our objects in the GUI) we need to store a string in the array of objects at the end to describe what the array contains.
		if ([obj isKindOfClass:[Object3D class]])
		{
			//Test to see if the ray intersects the object and that it is the closes object
			if ([obj intersects:ray] && ([ray t] < nearestDistance))
			{
				intersected = YES;
				nearestDistance = [ray t];	//Set the ray distance here
				//Set up our shading object
				[shader setObject:obj];
				[shader setIntersection:[obj hitPoint]];
				[shader setNormal:[obj normalAtPoint:[shader intersection]]];
				[shader setRay:ray];
			}
		}
	}
	return intersected;
}

//Calculates the reflected ray direction of the incident ray and the intersection
- (Vector3D *)reflectedRayDirection:(Ray *)incidentRay shader:(ShadingObject *)shader
{
	Vector3D *I = [[[Vector3D alloc] initWithVector:[incidentRay direction]] autorelease];
	return [I subtractVector:[[shader normal] multiplyByFloat:(2 * [I dotProduct:[shader normal]]) ]];
}

// Calculates the transmitted ray direction
- (Vector3D *)transmittedRayDirection:(Ray *)incidentRay shader:(ShadingObject *)shader
{
	Vector3D *I = [[Vector3D alloc] initWithVector:[[incidentRay direction] multiplyByFloat:-1.0]];
	float eta, cosTheta2, totIntRef;
	float cosTheta1;
	cosTheta1 = [I dotProduct:[shader normal]];
	[shader setKilled:NO];
	
	// if costheta > 0 we are in air, else we are in the intersected object. We do not handle CSG.
	if (cosTheta1 >= 0)
	{
		eta = [[[shader object] material] refractiveIndex];
		[shader setInside:NO];
	}
	else
	{
		cosTheta1 = cosTheta1 * -1;
		eta = 1/[[[shader object] material] refractiveIndex];
		[shader setNormal:[[shader normal] multiplyByFloat:-1.0]];
		[shader setInside:YES];
	}
	totIntRef = 1 - ((1/(eta * eta)) * (1 - (cosTheta1 * cosTheta1)));
	// if this is < 0, we get total internal reflection and need to kill the ray
	if (totIntRef < 0)
		[shader setKilled:YES];

	cosTheta2 = sqrt(totIntRef);
	[I release];
	return [[[incidentRay direction] multiplyByFloat:(1/eta)] subtractVector:[[shader normal] multiplyByFloat:(cosTheta2 - ((1/eta) * cosTheta1))]];
}

//These methods add objects to the scene.
- (void)addSphere
{
	Sphere *newSphere = [[Sphere alloc] init];
	[newSphere setObjectName:@"New Sphere"];
	[newSphere setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newSphere atIndex:[threeDObjects count]-1];
	[newSphere release];
}

- (void)addPlane
{
	Plane *newPlane = [[Plane alloc] init];
	[newPlane setObjectName:@"New Plane"];
	[newPlane setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newPlane atIndex:[threeDObjects count]-1];
	[newPlane release];
}

- (void)addLight
{
	Light *newLight = [[Light alloc] init];
	[newLight setName:@"New Light"];
	//Insert the object before the string @"Lights"
	[lights insertObject:newLight atIndex:[lights count]-1];
	[newLight release];
}

- (void)addTriangle
{
	Triangle *newTriangle = [[Triangle alloc] init];
	[newTriangle setObjectName:@"New Triangle"];
	[newTriangle setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newTriangle atIndex:[threeDObjects count]-1];
	[newTriangle release];
}

- (void)addGenericBox
{
	GenericBox *newGenericBox = [[GenericBox alloc] init];
	[newGenericBox setObjectName:@"New Generic Box"];
	[newGenericBox setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newGenericBox atIndex:[threeDObjects count]-1];
	[newGenericBox release];
}

- (void)addGenericSphere
{
	GenericSphere *newGenericSphere = [[GenericSphere alloc] init];
	[newGenericSphere setObjectName:@"New Generic Sphere"];
	[newGenericSphere setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newGenericSphere atIndex:[threeDObjects count]-1];
	[newGenericSphere release];
}

- (void)addGenericTorus
{
	GenericTorus *newGenericTorus = [[GenericTorus alloc] init];
	[newGenericTorus setObjectName:@"New Generic Torus"];
	[newGenericTorus setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newGenericTorus atIndex:[threeDObjects count]-1];
	[newGenericTorus release];
}

- (void)addGenericTaperedCylinder
{
	GenericTaperedCylinder *newGenericTaperedCylinder = [[GenericTaperedCylinder alloc] init];
	[newGenericTaperedCylinder setObjectName:@"New Generic Tapered Cylinder"];
	[newGenericTaperedCylinder setMaterial:[materials objectAtIndex:0]];
	//Insert the object before the string @"Objects"
	[threeDObjects insertObject:newGenericTaperedCylinder atIndex:[threeDObjects count]-1];
	[newGenericTaperedCylinder release];
}

@end
